<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-10
 ******************************************************************************/

namespace Fastmag\Sync\Logger;

use Fastmag\Sync\Api\LogRepositoryInterface;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\LogFactory;
use Magento\Framework\Exception\CouldNotSaveException;
use Monolog\Handler\HandlerInterface;
use Monolog\Logger as MonologLogger;

/**
 * Class Logger
 *
 * Dedicated logger class
 */
class Logger extends MonologLogger
{
    /** @var Config */
    protected $config;

    /** @var LogFactory */
    protected $logFactory;

    /** @var LogRepositoryInterface */
    protected $logRepository;

    /**
     * Logger constructor.
     *
     * @param string                 $name
     * @param Config                 $config
     * @param LogFactory             $logFactory
     * @param LogRepositoryInterface $logRepository
     * @param HandlerInterface[]     $handlers
     * @param callable[]             $processors
     */
    public function __construct(
        $name,
        Config $config,
        LogFactory $logFactory,
        LogRepositoryInterface $logRepository,
        array $handlers = [],
        array $processors = []
    ) {
        parent::__construct($name, $handlers, $processors);

        $this->config = $config;
        $this->logFactory = $logFactory;
        $this->logRepository = $logRepository;
    }

    /**
     * Log only ERROR, CRITICAL, ALERT, EMERGENCY level log if log disabled in configuration
     * and log in DB
     *
     * @inheritDoc
     */
    public function debug($message, array $context = []): void
    {
        if ($this->config->isSetFlag(Config::XML_PATH_LOG_ENABLE)) {
            if ($this->config->isSetFlag(Config::XML_PATH_LOG_TO_FILE)) {
                parent::debug($message, $context);
            }

            $this->logInDb(MonologLogger::DEBUG, $message);
        }
    }

    /**
     * Log only ERROR, CRITICAL, ALERT, EMERGENCY level log if log disabled in configuration
     * and log in DB
     *
     * @inheritDoc
     */
    public function info($message, array $context = []): void
    {
        if ($this->config->isSetFlag(Config::XML_PATH_LOG_ENABLE)) {
            if ($this->config->isSetFlag(Config::XML_PATH_LOG_TO_FILE)) {
                parent::info($message, $context);
            }

            $this->logInDb(MonologLogger::INFO, $message);
        }
    }

    /**
     * Log only ERROR, CRITICAL, ALERT, EMERGENCY level log if log disabled in configuration
     * and log in DB
     *
     * @inheritDoc
     */
    public function notice($message, array $context = []): void
    {
        if ($this->config->isSetFlag(Config::XML_PATH_LOG_ENABLE)) {
            if ($this->config->isSetFlag(Config::XML_PATH_LOG_TO_FILE)) {
                parent::notice($message, $context);
            }

            $this->logInDb(MonologLogger::NOTICE, $message);
        }
    }

    /**
     * Log only ERROR, CRITICAL, ALERT, EMERGENCY level log if log disabled in configuration
     * and log in DB
     *
     * @inheritDoc
     */
    public function warning($message, array $context = []): void
    {
        if ($this->config->isSetFlag(Config::XML_PATH_LOG_ENABLE)) {
            if ($this->config->isSetFlag(Config::XML_PATH_LOG_TO_FILE)) {
                parent::warning($message, $context);
            }

            $this->logInDb(MonologLogger::WARNING, $message);
        }
    }

    /**
     * Log in DB
     *
     * @inheritDoc
     */
    public function error($message, array $context = []): void
    {
        if ($this->config->isSetFlag(Config::XML_PATH_LOG_TO_FILE)) {
            parent::error($message, $context);
        }

        $this->logInDb(MonologLogger::ERROR, $message);
    }

    /**
     * Log in DB
     *
     * @inheritDoc
     */
    public function critical($message, array $context = []): void
    {
        if ($this->config->isSetFlag(Config::XML_PATH_LOG_TO_FILE)) {
            parent::critical($message, $context);
        }

        $this->logInDb(MonologLogger::CRITICAL, $message);
    }

    /**
     * Log in DB
     *
     * @inheritDoc
     */
    public function alert($message, array $context = []): void
    {
        if ($this->config->isSetFlag(Config::XML_PATH_LOG_TO_FILE)) {
            parent::alert($message, $context);
        }

        $this->logInDb(MonologLogger::ALERT, $message);
    }

    /**
     * Log in DB
     *
     * @inheritDoc
     */
    public function emergency($message, array $context = []): void
    {
        if ($this->config->isSetFlag(Config::XML_PATH_LOG_TO_FILE)) {
            parent::emergency($message, $context);
        }

        $this->logInDb(MonologLogger::EMERGENCY, $message);
    }

    /**
     * Log error in DB
     *
     * @param int    $level
     * @param string $message
     *
     * @return void
     */
    protected function logInDb(int $level, string $message): void
    {
        $log = $this->logFactory->create();
        $log->setMessage($message)
            ->setLevel($level);

        try {
            $this->logRepository->save($log);
        } catch (CouldNotSaveException $exception) {
            $this->addRecord(MonologLogger::CRITICAL, 'Unable to save log in DB, please review your installation');
        }
    }
}
