<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-06-19
 ******************************************************************************/

namespace Fastmag\Sync\Model\Config\Source\Rule;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Data\OptionSourceInterface;
use Magento\Shipping\Model\Config;
use Magento\Store\Model\ScopeInterface;

/**
 * Class ShippingMethods
 *
 * Source model getting all shipping methods available in Magento
 */
class ShippingMethods implements OptionSourceInterface
{
    /** @var ScopeConfigInterface $scopeConfig */
    protected $scopeConfig;

    /** @var Config */
    protected $shippingConfig;

    /** @var string[] $shippingMethods */
    protected $shippingMethods = [];

    /**
     * ShippingMethods constructor
     *
     * @param ScopeConfigInterface $scopeConfig
     * @param Config               $shippingConfig
     */
    public function __construct(ScopeConfigInterface $scopeConfig, Config $shippingConfig)
    {
        $this->scopeConfig = $scopeConfig;
        $this->shippingConfig = $shippingConfig;
    }

    /**
     * @inheritDoc
     */
    public function toOptionArray()
    {
        $options = [];

        foreach ($this->getShippingMethods() as $methodCode => $methodLabel) {
            $options[] = [
                'value' => $methodCode,
                'label' => $methodLabel
            ];
        }

        return $options;
    }

    /**
     * Returns values as associative array
     *
     * @return array
     */
    public function toArray()
    {
        return $this->getShippingMethods();
    }

    /**
     * Get shipping methods list
     *
     * @return array
     */
    protected function getShippingMethods()
    {
        if (count($this->shippingMethods) === 0) {
            $shippingMethods = ['*' => __('All shipping methods')];

            $carriers = $this->shippingConfig->getAllCarriers();
            foreach ($carriers as $carrierCode => $carrierModel) {
                $carrierMethods = $carrierModel->getAllowedMethods();
                if (!$carrierMethods) {
                    continue;
                }

                $carrierTitle = $this->scopeConfig->getValue(
                    'carriers/' . $carrierCode . '/title',
                    ScopeInterface::SCOPE_STORE
                );

                foreach ($carrierMethods as $methodCode => $methodTitle) {
                    if (!$methodCode) {
                        continue;
                    }

                    $shippingMethods[$methodCode] = $methodTitle . ' (' . $carrierTitle . ')';
                }
            }

            $this->shippingMethods = $shippingMethods;
        }

        return $this->shippingMethods;
    }
}
