<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-11-04
 ******************************************************************************/

namespace Fastmag\Sync\Model\Jobqueue;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterfaceFactory as JobFactory;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagSearchResultsInterfaceFactory as SearchResultsFactory;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface;
use Fastmag\Sync\Model\Jobqueue as AbstractJob;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToFastmag as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToFastmag\CollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface as CollectionProcessor;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SortOrder;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Zend_Db_Select;

/**
 * Class ToFastmagRepository
 *
 * Repository of jobs used for Magento to Fastmag synchronization
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class ToFastmagRepository extends StandardRepository implements ToFastmagRepositoryInterface
{
    /** @var ResourceModel $resource */
    protected $resource;

    /** @var JobFactory $jobFactory */
    protected $jobFactory;

    /** @var CollectionFactory $collectionFactory */
    protected $collectionFactory;

    /** @var SearchResultsFactory $searchResultsFactory */
    protected $searchResultsFactory;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /** @var CollectionProcessor $collectionProcessor */
    private $collectionProcessor;

    /**
     * ToFastmagRepository constructor
     *
     * @param ResourceModel            $resource
     * @param JobFactory               $jobFactory
     * @param CollectionFactory        $collectionFactory
     * @param SearchResultsFactory     $searchResultsFactory
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param SortOrderBuilder         $sortOrderBuilder
     * @param CollectionProcessor|null $collectionProcessor
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        ResourceModel $resource,
        JobFactory $jobFactory,
        CollectionFactory $collectionFactory,
        SearchResultsFactory $searchResultsFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder,
        CollectionProcessor $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->jobFactory = $jobFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(Job $job)
    {
        try {
            $oldJob = $this->checkExistingJobs($job);

            if ($oldJob !== false) {
                $oldJob->setProcessedAt(null)
                    ->setStatus(AbstractJob::STATUS_PENDING)
                    ->setMessage(null)
                    ->setTrace(null);

                $job = $oldJob;
            }

            $this->resource->save($job);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getById($jobId)
    {
        $job = $this->jobFactory->create();
        $this->resource->load($job, $jobId);
        if (!$job->getId()) {
            throw new NoSuchEntityException(__('The job with the "%1" ID doesn\'t exist.', $jobId));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function getListByCode(SearchCriteriaInterface $searchCriteria, $limit)
    {
        $searchResultsArray = [];

        $prioritiesCollection = $this->collectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $prioritiesCollection);

        $prioritiesCollection->getSelect()->reset(Zend_Db_Select::COLUMNS)
            ->columns(Job::JOB_CODE)
            ->distinct();

        /** @var Job $row */
        foreach ($prioritiesCollection as $row) {
            $collection = $this->collectionFactory->create();

            $this->searchCriteriaBuilder->setFilterGroups((array)$searchCriteria->getFilterGroups())
                ->addFilter(Job::JOB_CODE, $row->getJobCode());

            $rowCriteria = $this->searchCriteriaBuilder->create();

            $this->collectionProcessor->process($rowCriteria, $collection);

            $collection->setOrder(Job::JOB_ID, SortOrder::SORT_ASC)
                ->setPageSize($limit)
                ->setCurPage(1);

            $searchResultsArray[$row->getJobCode()] = $collection;
        }

        return $searchResultsArray;
    }

    /**
     * @inheritDoc
     */
    public function delete(Job $job)
    {
        try {
            $this->resource->delete($job);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($jobId)
    {
        return $this->delete($this->getById($jobId));
    }

    /**
     * @inheritDoc
     */
    public function reset($job)
    {
        parent::reset($job);

        $job->setRetryCount($job->getRetryCount() + 1);

        return $job;
    }

    /**
     * Check if a job already exists for the job code and content ID and returns it.
     * If there is more than one, a security has been added to delete the extra jobs
     *
     * @param Job $job
     *
     * @return Job|bool
     *
     * @throws CouldNotDeleteException
     * @throws NoSuchEntityException
     */
    protected function checkExistingJobs($job)
    {
        $result = false;

        $sortOrder = $this->sortOrderBuilder->setField(Job::CREATED_AT)
            ->setAscendingDirection()
            ->create();

        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(Job::CONTENT_ID, $job->getContentId())
            ->addFilter(Job::JOB_CODE, $job->getJobCode())
            ->addFilter(Job::JOB_ID, $job->getId(), 'neq')
            ->addSortOrder($sortOrder)
            ->create();

        $jobsList = $this->getList($searchCriteria);

        if ($jobsList->getTotalCount() > 1) {
            $result = $this->deleteExtraJobs($jobsList->getItems());
        } elseif ($jobsList->getTotalCount() === 1) {
            $jobsListAsArray = $jobsList->getItems();
            $result = reset($jobsListAsArray);
        }

        return $result;
    }

    /**
     * Delete extra jobs if there is more than one job for a specific content ID
     * Returns the first one
     *
     * @param Job[] $jobsList
     *
     * @return Job
     *
     * @throws CouldNotDeleteException
     * @throws NoSuchEntityException
     */
    protected function deleteExtraJobs($jobsList)
    {
        $jobsListCount = count($jobsList);

        if ($jobsListCount > 1) {
            for ($i = 1; $i < $jobsListCount; $i++) {
                $oldJob = $jobsList[$i];
                $this->deleteById($oldJob->getId());
            }
        }

        return $jobsList[0];
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessor
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
