<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-06-19
 ******************************************************************************/

namespace Fastmag\Sync\Model;

use Exception;
use Fastmag\Sync\Api\Data\LogApiInterface;
use Fastmag\Sync\Api\Data\LogApiInterfaceFactory;
use Fastmag\Sync\Api\Data\LogApiSearchResultsInterfaceFactory as SearchResultsFactory;
use Fastmag\Sync\Api\LogApiRepositoryInterface;
use Fastmag\Sync\Model\ResourceModel\LogApi as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\LogApi\CollectionFactory;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface as CollectionProcessor;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Reflection\DataObjectProcessor;

/**
 * Class LogApiRepository
 *
 * Repository class for log model
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class LogApiRepository implements LogApiRepositoryInterface
{
    /** @var ResourceModel $resource */
    protected $resource;

    /** @var LogApiFactory $logFactory */
    protected $logFactory;

    /** @var CollectionFactory $collectionFactory */
    protected $collectionFactory;

    /** @var SearchResultsFactory $searchResultsFactory */
    protected $searchResultsFactory;

    /** @var DataObjectHelper $dataObjectHelper */
    protected $dataObjectHelper;

    /** @var DataObjectProcessor $dataObjectProcessor */
    protected $dataObjectProcessor;

    /** @var LogApiInterfaceFactory $logInterfaceFactory */
    protected $logInterfaceFactory;

    /** @var CollectionProcessor $collectionProcessor */
    private $collectionProcessor;

    /**
     * LogRepository constructor
     *
     * @param ResourceModel            $resource
     * @param LogApiFactory            $logFactory
     * @param LogApiInterfaceFactory   $logInterfaceFactory
     * @param CollectionFactory        $collectionFactory
     * @param SearchResultsFactory     $searchResultsFactory
     * @param DataObjectHelper         $dataObjectHelper
     * @param DataObjectProcessor      $dataObjectProcessor
     * @param CollectionProcessor|null $collectionProcessor
     */
    public function __construct(
        ResourceModel          $resource,
        LogApiFactory          $logFactory,
        LogApiInterfaceFactory $logInterfaceFactory,
        CollectionFactory      $collectionFactory,
        SearchResultsFactory   $searchResultsFactory,
        DataObjectHelper       $dataObjectHelper,
        DataObjectProcessor    $dataObjectProcessor,
        CollectionProcessor    $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->logFactory = $logFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->logInterfaceFactory = $logInterfaceFactory;
        $this->dataObjectProcessor = $dataObjectProcessor;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(LogApiInterface $log): LogApiInterface
    {
        try {
            $this->resource->save($log);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $log;
    }

    /**
     * @inheritDoc
     */
    public function getById($logId)
    {
        $log = $this->logFactory->create();
        $this->resource->load($log, $logId);
        if (!$log->getId()) {
            throw new NoSuchEntityException(__('The API call log with the "%1" ID doesn\'t exist.', $logId));
        }

        return $log;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(LogApiInterface $log): bool
    {
        try {
            $this->resource->delete($log);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($logId)
    {
        return $this->delete($this->getById($logId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessor
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
