<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-06-19
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration;

use Exception;
use Fastmag\Sync\Api\Data\OrderInterfaceFactory as SyncedOrderFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\OrderRepositoryInterface as SyncedOrderRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\FastmagSql as FastmagSqlTrait;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration as IntegrationTrait;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Order
 *
 * Abstract class for Order related Integration workers
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
abstract class Order extends Worker
{
    use IntegrationTrait;
    use FastmagSqlTrait;

    /** @var Config $config */
    protected $config;

    /** @var SyncedOrderFactory $syncedOrderFactory */
    protected $syncedOrderFactory;

    /** @var SyncedOrderRepository $syncedOrderRepository */
    protected $syncedOrderRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @var OrderEntity $currentEntity */
    protected $currentEntity;

    /** @var array $currentJobApiCalls */
    protected $currentJobApiCalls = [];

    /**
     * Order constructor
     *
     * @param Logger                $logger
     * @param JobRepository         $jobRepository
     * @param Api                   $api
     * @param Config                $config
     * @param SyncedOrderFactory    $syncedOrderFactory
     * @param SyncedOrderRepository $syncedOrderRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param Json                  $jsonSerializer
     * @param Proxy                 $fastmagSql
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        Config $config,
        SyncedOrderFactory $syncedOrderFactory,
        SyncedOrderRepository $syncedOrderRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Json $jsonSerializer,
        Proxy $fastmagSql
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->api = $api;
        $this->config = $config;
        $this->syncedOrderFactory = $syncedOrderFactory;
        $this->syncedOrderRepository = $syncedOrderRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jsonSerializer = $jsonSerializer;
        $this->fastmagSql = $fastmagSql;
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_ORDER_EXPORT_ENABLE);
    }

    /**
     * Send cancellation to Fastmag
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendCancellation()
    {
        $objectName = 'Ventes';
        $endpoint = '/boa/sale/cancel/index.ips';

        $type = $this->currentEntity->getTransactionType();
        if ($type === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION_CANCELLATION) {
            $objectName = 'Reservations';
            $endpoint = '/boa/Reservation/cancel/index.ips';
        }

        if ($this->currentEntity->getLastTransaction() === null) {
            throw new JobException(__(
                'Unable to cancel order #%1 to Fastmag through EDI: no previous transaction found.',
                $this->currentEntity->getMagentoId()
            ));
        }

        $lastTransactionId = $this->currentEntity->getLastTransaction()->getTransactionId();

        $request = (object)[$objectName => [(object)['Numero' => $lastTransactionId]]];

        try {
            $this->sendRequest($type, $endpoint, $request, $lastTransactionId);
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to cancel order #%1 on Fastmag through API. Code: %2. Message: %3. Cause: %4. Data sent: %5',
                $this->currentEntity->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                $exception->getRequest()
            ));
        }
    }

    /**
     * Send request to Fastmag through API
     *
     * @param string   $transactionType
     * @param string   $endpoint
     * @param mixed    $request
     * @param int|null $transactionId
     *
     * @return void
     *
     * @throws ApiException
     * @throws JobException
     */
    protected function sendRequest($transactionType, $endpoint, $request, $transactionId = null)
    {
        $callData = [
            'transaction_id'   => $transactionId,
            'transaction_type' => $transactionType,
            'endpoint'         => $endpoint,
            'request'          => $this->jsonSerializer->serialize($request),
            'message'          => null,
            'request_at'       => date('Y-m-d H:i:s'),
            'origin_shop'      => null
        ];

        if ($transactionType === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE) {
            $callData['transaction_id'] = $request->Reservations[0]->NumResa;
        }

        $response = $this->api->post($endpoint, $request);

        if ($transactionType !== OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE
            && array_key_exists('data', $response)
        ) {
            $response = $response['data'][0];

            if ($response['status'] === 'OK') {
                $callData['transaction_id'] = $response['Id'];
            }
        }

        $callData['status'] = $response['status'];
        $callData['response_at'] = date('Y-m-d H:i:s');

        if ($response['status'] === 'KO') {
            $message = (array_key_exists('Message', $response) ? $response['Message'] : '');
            if (preg_match('/l\'InfosComp de la vente \d* existe déjà/iu', $message) === 1) {
                $callData['status'] = 'OK';
                $callData['transaction_id'] = $this->getPreviousTransactionId();
            } else {
                throw new ApiException(
                    __($message),
                    $this->api->getLastRequest(),
                    null,
                    (array_key_exists('errorCode', $response) ? $response['errorCode'] : 0)
                );
            }

        }

        $this->currentJobApiCalls[] = $callData;
    }

    /**
     * Get previous transaction ID for current order
     * Used only if there was a timeout in the previous current sale creation request
     *
     * @return int
     *
     * @throws JobException
     */
    protected function getPreviousTransactionId()
    {
        $incrementId = $this->currentEntity->getIncrementId();

        try {
            $sql = 'SELECT COUNT(*) AS previous_transactions
                FROM ventes
                WHERE InfosComp LIKE ' . $this->getFastmagSqlConnection()->escape('%' . $incrementId);

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if ((int)$rows[0]['previous_transactions'] === 0) {
                throw new JobException(__(
                    'Order #%1 has already been synced one time in Fastmag, '
                    . 'but we\'re unable to find the matching transaction. '
                    . 'Please review it in Fastmag to remove duplicate sales'
                ));
            }

            if ((int)$rows[0]['previous_transactions'] > 1) {
                throw new JobException(__(
                    'Order #%1 has been synced more than one time in Fastmag. '
                    . 'Please review them in Fastmag to remove duplicate sales'
                ));
            }

            $sql = 'SELECT Vente AS transaction_id
                FROM ventes
                WHERE InfosComp LIKE ' . $this->getFastmagSqlConnection()->escape('%' . $incrementId);

            $rows = $this->getFastmagSqlConnection()->get($sql);

            $result = (int)$rows[0]['transaction_id'];
        } catch (NoConnectionException $exception) {
            throw new JobException(__('Unable to connect to Fastmag. Cause: ' . $exception->getMessage()));
        }

        return $result;
    }

    /**
     * Save order synchronizations
     *
     * @param string $requestAt
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveOrderSync($requestAt)
    {
        foreach ($this->currentJobApiCalls as $apiCall) {
            $syncedOrder = $this->syncedOrderFactory->create();

            $syncedOrder->setOrderId($this->currentEntity->getMagentoId())
                ->setTransactionId($apiCall['transaction_id'])
                ->setTransactionType($apiCall['transaction_type'])
                ->setRequest($apiCall['request'])
                ->setEndpoint($apiCall['endpoint'])
                ->setStatus($apiCall['status'])
                ->setRequestAt($requestAt)
                ->setResultAt($apiCall['response_at'])
                ->setOriginShop($apiCall['origin_shop'])
                ->setTargetShop($this->currentEntity->getFastmagShop());

            try {
                $this->syncedOrderRepository->save($syncedOrder);
            } catch (Exception $exception) {
                throw new JobException(
                    __('Unable to save order #%1 synchronization on Magento DB', $this->currentEntity->getMagentoId())
                );
            }
        }
    }
}
