<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2023 HOMEMADE.IO SAS
 * @date      2023-09-04
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProductNotSyncableException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;

/**
 * Class Price
 *
 * Hydration class used for inserting or updating products prices from Fastmag to Magento
 */
class Price extends Product
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product_price';

    /** @inheritDoc */
    protected $syncConfigFields = [
        Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS,
        Config::XML_PATH_PRICE_IMPORT_FASTMAG_RATE,
        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_REFERENCE_STOCK,
        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS,
        Config::XML_PATH_ORDER_WORKFLOW_SHOP
    ];

    /**
     * @inheritDoc
     */
    public function run()
    {
        $this->getSyncConfig();

        foreach ($this->getJobs()->getItems() as $job) {
            $this->currentJob = $job;

            $fastmagRef = $this->currentJob->getContentId();

            if ($this->getMagentoProductId($fastmagRef) !== false) {
                try {
                    $entity = $this->getDataFromFastmag();

                    $this->hydrateJob($job, $entity);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                } catch (ProductNotSyncableException $exception) {
                    $this->skipJob($job, $exception);
                }
            } else {
                $exception = new ProductNotSyncableException(__(
                    'Can not change prices of the product with Fastmag ref "%1", as it does not exist in Magento',
                    $fastmagRef
                ));

                $this->skipJob($job, $exception);
            }

            $this->saveJob($job);
        }
    }

    /**
     * @inheritDoc
     */
    protected function getParentData()
    {
        try {
            $sql = 'SELECT pf.VisibleWeb AS visible_web,
                    pf.Actif AS active,
                    COUNT(s.ID) AS stock_level,
                    f.Famille AS vat_family,
                    f.TauxTVA AS vat_rate
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode =
                    ' . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                    AND s.AR = 1
                GROUP BY pf.BarCode';

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) === 0) {
                $sql = 'SELECT pf.VisibleWeb AS visible_web,
                    pf.Actif AS active,
                    0 AS stock_level,
                    f.Famille AS vat_family,
                    f.TauxTVA AS vat_rate
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode = '
                    . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                GROUP BY pf.BarCode';

                $rows = $this->getFastmagSqlConnection()->get($sql);
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        $row = reset($rows);

        $this->currentEntity->addData($row);
    }

    /**
     * @inheritDoc
     */
    protected function getChildrenDataSql($storeId = null)
    {
        $fastmagShop = $this->getSyncConfigValueFromArray(Config::XML_PATH_ORDER_WORKFLOW_SHOP, $storeId);
        $rateCode = $this->getSyncConfigValueFromArray(Config::XML_PATH_PRICE_IMPORT_FASTMAG_RATE, $storeId);

        $standardPriceSubquery = $this->getStandardPriceSubquery($fastmagShop, $rateCode);
        $combinationPriceSubquery = $this->getCombinationPriceSubquery($rateCode);
        $ratePriceSubquery = $this->getRatePriceSubquery($rateCode);
        $salePriceSubquery = $this->getSalePriceSubquery($fastmagShop);
        $variationSalePriceSubquery = $this->getVariationSalePriceSubquery($fastmagShop);
        $buyingPriceSubquery = $this->getBuyingPriceSubquery();

        return 'SELECT
                (
                    SELECT 1
                    FROM stock AS s2
                    WHERE s2.BarCode = stock.BarCode AND s2.Taille = stock.Taille AND s2.Couleur = stock.Couleur
                        ' . $this->getStocksCodesCondition($storeId) . '
                    LIMIT 1
                ) AS active_stock,
                pf.PrixVente AS indicative_price,
                (' . $standardPriceSubquery . ') AS standard_price,
                (' . $combinationPriceSubquery . ') AS combination_price,
                (' . $ratePriceSubquery . ') AS rate_price,
                (' . $salePriceSubquery . ') AS sale_price,
                (' . $variationSalePriceSubquery . ') AS variation_sale_price,
                (' . $buyingPriceSubquery . ') AS buying_price,
                TRIM(stock.Taille) AS size,
                TRIM(stock.Couleur) AS color,
                (
                    SELECT TRIM(Descriptif)
                    FROM produitscouleurs AS pc
                    WHERE pc.Barcode = stock.BarCode AND pc.Couleur = stock.Couleur
                ) AS color_description
            FROM produitsfiches AS pf INNER JOIN stock ON pf.BarCode = stock.BarCode
            WHERE stock.AR = 1
                AND stock.BarCode =
                ' . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
            GROUP BY stock.Barcode, stock.Taille, stock.Couleur
            ' . $this->getChildrenFilters() . '
            ORDER BY stock.Couleur, stock.Taille';
    }
}
