<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2023 HOMEMADE.IO SAS
 * @date      2023-03-01
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer\Address;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Address as AbstractAddress;
use Magento\Customer\Api\AddressRepositoryInterface as AddressRepository;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Shipping
 *
 * Hydration class used for inserting or updating customer shipping addresses from Magento to Fastmag
 */
class Shipping extends AbstractAddress
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_customer_address_shipping';

    /** @var CustomerEntityFactory $customerEntityFactory */
    protected $customerEntityFactory;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /**
     * Shipping constructor
     *
     * @param Logger                $logger
     * @param JobRepository         $jobRepository
     * @param Config                $config
     * @param Json                  $jsonSerializer
     * @param CustomerRepository    $customerRepository
     * @param AddressRepository     $addressRepository
     * @param AddressEntityFactory  $addressEntityFactory
     * @param CustomerEntityFactory $customerEntityFactory
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        CustomerRepository $customerRepository,
        AddressRepository $addressRepository,
        AddressEntityFactory $addressEntityFactory,
        CustomerEntityFactory $customerEntityFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
        parent::__construct(
            $logger,
            $jobRepository,
            $config,
            $jsonSerializer,
            $customerRepository,
            $addressRepository,
            $addressEntityFactory
        );

        $this->customerEntityFactory = $customerEntityFactory;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        $entities = $this->getDataFromMagento();

        if (is_array($entities)) {
            foreach ($this->getJobs()->getItems() as $job) {
                if (array_key_exists($job->getContentId(), $entities)) {
                    /** @var CustomerEntity $customerEntity */
                    $customerEntity = $entities[$job->getContentId()];

                    if ($customerEntity->getShippingAddress() === null) {
                        $exception = new JobException(__('No matching address found for job'));
                        $this->invalidateJob($job, $exception);
                    } else {
                        $this->hydrateJob($job, $customerEntity);
                    }
                } else {
                    $exception = new JobException(__('No matching customer found for job'));
                    $this->invalidateJob($job, $exception);
                }

                $this->saveJob($job);
            }
        }
    }

    /**
     * @inheritDoc
     */
    protected function getDataFromMagento()
    {
        $customersIds = $this->getJobs()->getColumnValues(Job::CONTENT_ID);
        $result = [];

        foreach ($customersIds as $customerId) {
            $customerEntity = $this->customerEntityFactory->create();

            try {
                $customer = $this->customerRepository->getById($customerId);
            } catch (Exception $exception) {
                continue;
            }

            $customerEntity->setMagentoId($customerId)
                ->setEmailAddress($customer->getEmail());

            $addressId = $customer->getDefaultShipping();

            if ($addressId) {
                try {
                    $addressObject = $this->addressRepository->getById($addressId);

                    $addressEntity = $this->generateAddressEntity(
                        $addressObject,
                        $this->getAddressAlias($addressObject)
                    );

                    $customerEntity->setShippingAddress($addressEntity);
                } catch (Exception $exception) {
                    continue;
                }
            }

            $result[$customerId] = $customerEntity;
        }

        return $result;
    }
}
