<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-05-16
 ******************************************************************************/

namespace Fastmag\Sync\Cron;

use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Model\Config;
use Laminas\Http\Exception\RuntimeException;
use Laminas\Http\Request;
use Laminas\Uri\Uri as UriHandler;
use Magento\Framework\App\ProductMetadataInterface as ProductMetadata;
use Magento\Framework\Exception\NoSuchEntityException;
use Laminas\Http\Client as HttpClient;
use Magento\Framework\Module\ResourceInterface as ModuleResource;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Class PushVersion
 *
 * Cron class used for log rotation
 */
class PushVersion
{
    /** @var Config $config */
    protected Config $config;

    /** @var HttpClient $httpClient */
    protected HttpClient $httpClient;

    /** @var ModuleResource $moduleResource */
    protected ModuleResource $moduleResource;

    /** @var StoreManagerInterface $storeManager */
    protected StoreManagerInterface $storeManager;

    /** @var UriHandler $uriHandler */
    protected UriHandler $uriHandler;

    /** @var ProductMetadata $productMetadata */
    protected ProductMetadata $productMetadata;

    /**
     * PushVersion constructor
     *
     * @param Config                $config
     * @param HttpClient            $httpClient
     * @param ModuleResource        $moduleResource
     * @param StoreManagerInterface $storeManager
     * @param UriHandler            $uriHandler
     * @param ProductMetadata       $productMetadata
     */
    public function __construct(
        Config                $config,
        HttpClient            $httpClient,
        ModuleResource        $moduleResource,
        StoreManagerInterface $storeManager,
        UriHandler            $uriHandler,
        ProductMetadata       $productMetadata
    ) {
        $this->config = $config;
        $this->httpClient = $httpClient;
        $this->moduleResource = $moduleResource;
        $this->storeManager = $storeManager;
        $this->uriHandler = $uriHandler;
        $this->productMetadata = $productMetadata;
    }

    /**
     * Send module version to monitoring website
     *
     * @return void
     *
     * @throws NoSuchEntityException|NoConnectionException
     */
    public function execute(): void
    {
        $websiteHmioId = $this->config->getValue(Config::XML_PATH_ABOUT_WEBSITE_ID);
        $customer = 'fastmag_' . strtolower($this->config->getValue(Config::XML_PATH_CONNECT_API_CHAIN));
        $url = $this->storeManager->getStore()->getBaseUrl();
        $urlParsed = $this->uriHandler->parse($url);

        try {
            $this->httpClient->setUri('https://www.home-made.io/monitoring/fastmagsync_modules_list.php')
                ->setMethod(Request::METHOD_POST)
                ->setHeaders(['Content-Type' => 'application/x-www-form-urlencoded'])
                ->setParameterPost([
                    'version'     => $this->moduleResource->getDbVersion('Fastmag_Sync'),
                    'cms'         => 'Magento 2',
                    'cms_version' => $this->productMetadata->getVersion(),
                    'website'     => $urlParsed->getScheme() . '://' .  $urlParsed->getHost(),
                    'id_site'     => $websiteHmioId,
                    'path'        => $urlParsed->getPath(),
                    'customer'    => $customer
                ]);

            $this->httpClient->send();
        } catch (RuntimeException $exception) {
            throw new NoConnectionException(__('Unable to push module version: %1', $exception->getMessage()));
        }

        $response = $this->httpClient->getResponse();

        if ($response && ($response->getStatusCode() < 200 || $response->getStatusCode() >= 300)) {
            throw new NoConnectionException(__(
                'Unable to push module version. Status: %1, message: %2',
                $response->getStatusCode(),
                $response->getReasonPhrase()
            ));
        }
    }
}
