<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2023 HOMEMADE.IO SAS
 * @date      2023-08-01
 ******************************************************************************/

namespace Fastmag\Sync\Process\Entity\ToMagento;

use DateTime;
use DateTimeZone;
use Exception;
use Fastmag\Sync\Helper\Data as Helper;
use Fastmag\Sync\Process\Entity\Date as DateTrait;
use Fastmag\Sync\Process\Entity\Helper\Attribute\Gender;
use Fastmag\Sync\Process\Entity\Helper\Attribute\Prefix;
use Fastmag\Sync\Process\Entity\ToMagento\Customer\Address;
use Fastmag\Sync\Process\EntityInterface;
use Laminas\Validator\EmailAddress as EmailAddressValidator;
use Magento\Framework\DataObject;

/**
 * Class Customer
 *
 * Entity class for customer.
 * Save and transform data from Fastmag for Magento easy integration,
 * and transfer it from hydration worker to integration worker
 *
 * @method int|null getFastmagId()
 * @method int|null getMagentoId()
 * @method int|null getBirthYear()
 * @method int|null getBirthMonth()
 * @method int|null getBirthDay()
 * @method string|null getComment()
 * @method Address|null getBillingAddress()
 * @method Address|null getShippingAddress()
 * @method int|null getMagentoDefaultBilling()
 * @method int|null getMagentoDefaultShipping()
 * @method Customer setFastmagId(int $fastmagId)
 * @method Customer setMagentoId(int $magentoId)
 * @method Customer setActive(bool $isActive)
 * @method Customer setGender(string $gender)
 * @method Customer setPrefix(string $prefix)
 * @method Customer setFirstname(string $firstname)
 * @method Customer setLastname(string $lastname)
 * @method Customer setEmailAddress(string $emailAddress)
 * @method Customer setBirthYear(int $birthYear)
 * @method Customer setBirthMonth(int $birthMonth)
 * @method Customer setBirthDay(int $birthDay)
 * @method Customer setDiscountRate(float $discountRate)
 * @method Customer setGrouping(string $grouping)
 * @method Customer setCreationDate(string $creationDate)
 * @method Customer setComment(string $comment)
 * @method Customer setBillingAddress(Address $address)
 * @method Customer setShippingAddress(Address $address)
 * @method Customer setMagentoDefaultBilling(int $addressId)
 * @method Customer setMagentoDefaultShipping(int $addressId)
 */
class Customer extends DataObject implements EntityInterface
{
    use DateTrait;

    /** @var EmailAddressValidator $emailAddressValidator */
    protected EmailAddressValidator $emailAddressValidator;

    /**
     * Product constructor
     *
     * @param EmailAddressValidator $emailAddressValidator
     * @param array                 $data
     */
    public function __construct(EmailAddressValidator $emailAddressValidator, array $data = [])
    {
        parent::__construct($data);

        $this->emailAddressValidator = $emailAddressValidator;

        $this->timeZoneInput = 'Europe/Paris';
        $this->timeZoneOutput = 'Europe/Paris';

        $this->configureDateFormatter();
    }

    /**
     * Get active
     *
     * @return bool|null
     */
    public function isActive()
    {
        return $this->getData('active') !== null ? (bool)$this->getData('active') : null;
    }

    /**
     * Get gender
     *
     * @return int|null
     */
    public function getGender()
    {
        return $this->getData('gender') !== null
            ? Gender::getMagentoValue($this->getData('gender'))
            : Gender::getDefaultMagentoValue();
    }

    /**
     * Get prefix
     *
     * @return string
     */
    public function getPrefix(): ?string
    {
        return $this->getData('prefix') !== null
            ? Prefix::getMagentoValue($this->getData('prefix'))
            : Prefix::getDefaultMagentoValue();
    }

    /**
     * Get firstname
     *
     * @return string
     */
    public function getFirstname(): ?string
    {
        $firstname = $this->getData('firstname');
        $lastname = $this->getData('lastname');

        if (empty($firstname) && !empty($lastname)) {
            $names = explode(' ', $lastname);
            $firstname = array_pop($names);
        }

        return Helper::capitalize($firstname);
    }

    /**
     * Get lastname
     *
     * @return string
     */
    public function getLastname(): ?string
    {
        $firstname = $this->getData('firstname');
        $lastname = $this->getData('lastname');

        if (empty($firstname) && !empty($lastname)) {
            $names = explode(' ', $lastname);
            array_pop($names);
            $lastname = implode(' ', $names);
        }

        return Helper::capitalize($lastname);
    }

    /**
     * Get email address
     *
     * @return string
     */
    public function getEmailAddress(): ?string
    {
        $result = false;

        $emailAddress = $this->getData('email_address');

        if ($this->emailAddressValidator->isValid($emailAddress)) {
            $result = Helper::lower($emailAddress);
        }

        return $result;
    }

    /**
     * Get date of birth
     *
     * @return string|null
     */
    public function getDob()
    {
        $result = null;

        $value = $this->getBirthYear() . '-' . $this->getBirthMonth() . '-' . $this->getBirthDay();

        if (!empty($value)) {
            if (!($value instanceof DateTime)) {
                try {
                    $this->setData('dob', new DateTime($value, new DateTimeZone($this->timeZoneInput)));
                } catch (Exception $exception) {
                    // Do nothing
                }
            }

            $this->dateFormatter->setPattern('y-MM-dd');
            $result = $this->dateFormatter->format($this->getData('dob'));
        }

        return $result;
    }

    /**
     * Get token newsletter subscription
     *
     * @return bool|null
     */
    public function getSubscribeNewsletter()
    {
        return $this->getData('subscribe_nl') !== null ? (bool)$this->getData('subscribe_nl') : null;
    }

    /**
     * Get discount rate
     *
     * @return float|null
     */
    public function getDiscountRate()
    {
        return (float)$this->getData('discount_rate') !== 0.0 ? (float)$this->getData('discount_rate') : null;
    }

    /**
     * Get grouping
     *
     * @return string|null
     */
    public function getGrouping()
    {
        return trim($this->getData('grouping')) !== '' ? trim($this->getData('grouping')) : null;
    }

    /**
     * Get creation date
     *
     * @return string
     */
    public function getCreationDate(): ?string
    {
        return $this->getFormattedDate('creation_date', 'y-MM-dd HH:mm:ss');
    }

    /**
     * Set token of newsletter subscription
     *
     * @param float $subscribeNewsletter
     *
     * @return Customer
     */
    public function setSubscribeNewsletter($subscribeNewsletter): Customer
    {
        return $this->setData('subscribe_nl', $subscribeNewsletter);
    }

    /**
     * @inheritDoc
     */
    public function export()
    {
        $result = [
            'fastmag_id'       => $this->getFastmagId(),
            'magento_id'       => $this->getMagentoId(),
            'is_active'        => $this->isActive(),
            'gender'           => $this->getGender(),
            'prefix'           => $this->getPrefix(),
            'firstname'        => $this->getFirstname(),
            'lastname'         => $this->getLastname(),
            'email_address'    => $this->getEmailAddress(),
            'dob'              => $this->getDob(),
            'subscribe_nl'     => $this->getSubscribeNewsletter(),
            'discount_rate'    => $this->getDiscountRate(),
            'grouping'         => $this->getGrouping(),
            'creation_date'    => $this->getCreationDate(),
            'comment'          => $this->getComment(),
            'billing_address'  => ($this->getBillingAddress() ? $this->getBillingAddress()->export() : null),
            'shipping_address' => ($this->getShippingAddress() ? $this->getShippingAddress()->export() : null),
        ];

        return array_filter(
            $result,
            static function ($value) {
                return $value !== null && $value !== [];
            }
        );
    }
}
