<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-06-19
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Order;

use Fastmag\Sync\Api\Data\OrderInterfaceFactory as SyncedOrderFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\OrderRepositoryInterface as SyncedOrderRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\Transfer as TransferEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Transfer;
use Magento\Framework\Serialize\Serializer\Json;
use stdClass;

/**
 * Class CancellationTransfer
 *
 * Integration worker used for reversing inventory transfers when an order is cancelled in Fastmag
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class CancellationTransfer extends Transfer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_order_cancellation_transfer';

    /** @var SyncedOrderFactory $syncedOrderFactory */
    protected $syncedOrderFactory;

    /** @var SyncedOrderRepository $syncedOrderRepository */
    protected $syncedOrderRepository;

    /** @inheritDoc */
    protected $transactionType = OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_TRANSFER_CANCELLATION;

    /**
     * CancellationTransfer constructor
     *
     * @param Logger                $logger
     * @param JobRepository         $jobRepository
     * @param Api                   $api
     * @param Config                $config
     * @param Json                  $jsonSerializer
     * @param SyncedOrderFactory    $syncedOrderFactory
     * @param SyncedOrderRepository $syncedOrderRepository
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        Config $config,
        Json $jsonSerializer,
        SyncedOrderFactory $syncedOrderFactory,
        SyncedOrderRepository $syncedOrderRepository
    ) {
        parent::__construct($logger, $jobRepository, $api, $config, $jsonSerializer);

        $this->syncedOrderFactory = $syncedOrderFactory;
        $this->syncedOrderRepository = $syncedOrderRepository;
    }

    /**
     * @inheritDoc
     */
    protected function processJob($job)
    {
        $this->currentJob = $job;

        if ($this->isTransactionCancellation($this->currentJob->getEntity()->getTransactionType())) {
            $transfers = $this->currentJob->getEntity()->getTransfers();

            foreach ($transfers as $transfer) {
                $this->sendCancellationTransfer($transfer);
            }
        }
    }

    /**
     * Check if the transaction is a sale or reservation cancellation
     *
     * @param string $transactionType
     *
     * @return bool
     */
    protected function isTransactionCancellation($transactionType)
    {
        return in_array($transactionType, [
            OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE_CANCELLATION,
            OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION_CANCELLATION
        ], true);
    }

    /**
     * Send cancellation transfer
     *
     * @param TransferEntity $transferEntity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendCancellationTransfer($transferEntity)
    {
        $endpoint = '/boa/transfert/create/index.ips';
        $request = $this->generateTransferDataRequest($transferEntity);

        try {
            $requestAt = date('Y-m-d H:i:s');

            $response = $this->api->post($endpoint, $request);
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send inventory transfer cancellation for order #%1 on Fastmag through API. Code: %2. Message: %3. Cause: %4. Data sent: %5',
                $this->currentJob->getEntity()->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                $exception->getRequest()
            ));
        }

        if (array_key_exists('data', $response)) {
            $response = $response['data'][0];
        }

        if ($response['status'] === 'OK') {
            $transferData = $request->Transferts[0];

            if ($this->config->isSetFlag(Config::XML_PATH_ORDER_ALERT_ENABLE_FOR_TRANSFER)
                && !$this->config->isSetFlag(Config::XML_PATH_ORDER_WORKFLOW_INVENTORY_TRANSFER)
            ) {
                $this->sendAlert(
                    $transferData->MagasinOrigine,
                    $transferData->MagasinDestination
                );
            }

            $this->saveSyncedTransfer(
                $request,
                $response,
                $requestAt,
                $endpoint,
                $transferData->MagasinOrigine,
                $transferData->MagasinDestination,
                Constants::TRANSFER_TYPE_CANCELLATION
            );
        } else {
            throw new JobException(__(
                'Unable to send inventory transfer cancellation for order #%1 on Fastmag through API. Message: %2. Data sent: %3',
                $this->currentJob->getEntity()->getMagentoId(),
                $response['Message'],
                $this->jsonSerializer->serialize($request)
            ));
        }
    }

    /**
     * Generate API request for transfer
     *
     * @param TransferEntity $transferEntity
     *
     * @return stdClass
     */
    protected function generateTransferDataRequest($transferEntity)
    {
        $transferRequest = $this->jsonSerializer->unserialize($transferEntity->getRequest());

        $transferData = $transferRequest['Transferts'][0];

        $transferData['MagasinOrigine'] = $transferEntity->getTargetShop();
        $transferData['MagasinDestination'] = $transferEntity->getOriginShop();

        return (object)['Transferts' => [(object)$transferData]];
    }

    /**
     * Send alert to shop
     *
     * @param string $senderShop
     * @param string $recipientShop
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendAlert($senderShop, $recipientShop)
    {
        $alertMessage = __('Some orders which needed inventory transfers have been cancelled. Backtransfers have been made, please consider validate them.');

        $request = (object)[
            'Consignes' => [(object)[
                'Destinataire' => $recipientShop,
                'Sujet'        => __('Backtransfers to validate'),
                'Message'      => $alertMessage,
                'Expediteur'   => $senderShop,
                'Slide'        => '1'
            ]]
        ];

        try {
            $response = $this->api->post('/boa/consigne/create/index.ips', $request);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(__($message), $this->api->getLastRequest(), null, $response['errorCode']);
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send alert to shop %1. Code: %2. Message: %3. Cause: %4. Alert sent: %5',
                $this->currentJob->getEntity()->getFastmagShop(),
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                $alertMessage
            ));
        }
    }
}
