<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-05-16
 ******************************************************************************/

namespace Fastmag\Sync\Model\System\Connection\Sql;

use Exception;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config\Provider\DirectSql as DirectSqlConfig;
use InvalidArgumentException;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\DB\Adapter\Pdo\Mysql as MysqlAdapter;
use Magento\Framework\DB\Adapter\Pdo\MysqlFactory as MysqlAdapterFactory;
use Magento\Framework\Model\ResourceModel\Type\Db\Pdo\Mysql as MysqlResource;
use PDO;
use Zend_Db_Adapter_Exception;

/**
 * Class DirectSql
 *
 * Class to connect and query to Fastmag through direct SQL interfaces
 */
class DirectSql extends MysqlResource implements SqlInterface
{
    /* @var MysqlAdapter */
    protected $connection;

    /** @var DirectSqlConfig */
    protected $directSqlConfigProvider;

    /* @var MysqlAdapterFactory */
    protected $connectionFactory;

    /** @var Logger */
    protected $logger;

    /** @var ScopeConfigInterface */
    protected $scopeConfig;

    /**
     * DirectSql constructor.
     *
     * @param DirectSqlConfig          $directSqlConfigProvider
     * @param Logger                   $logger
     * @param ScopeConfigInterface     $scopeConfig
     * @param MysqlAdapterFactory|null $mysqlFactory
     */
    public function __construct(
        DirectSqlConfig $directSqlConfigProvider,
        Logger $logger,
        ScopeConfigInterface $scopeConfig,
        MysqlAdapterFactory $mysqlFactory = null
    ) {
        $this->directSqlConfigProvider = $directSqlConfigProvider;
        $this->logger = $logger;
        $this->scopeConfig = $scopeConfig;

        $config = $this->getCredentialsFromConfig();

        parent::__construct($config, $mysqlFactory);
    }

    /**
     * @inheritDoc
     */
    public function connect()
    {
        if ($this->connection === null) {
            $this->connection = $this->getConnection();
        }

        try {
            /** @throws Zend_Db_Adapter_Exception */
            $this->connection->getServerVersion();
        } catch (Zend_Db_Adapter_Exception $exception) {
            throw new NoConnectionException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function get($sql)
    {
        try {
            $result = $this->connection->query($sql);
        } catch (Exception $exception) {
            throw new NoConnectionException(__($exception->getMessage()));
        }

        return $result === null ? [] : $result->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * @inheritDoc
     */
    public function post($sql)
    {
        try {
            $result = $this->connection->query($sql) !== null;
        } catch (Exception $exception) {
            throw new NoConnectionException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function multiPost($sql)
    {
        try {
            $result = $this->connection->multiQuery($sql) !== null;
        } catch (Exception $exception) {
            throw new NoConnectionException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function escape($value, $forceString = false)
    {
        $result = $value;

        if (is_array($value)) {
            $result = '';

            foreach ($value as $row) {
                $result .= $this->escape($row, $forceString) . ', ';
            }

            $result = trim($result, ', ');
        } elseif ($value !== null) {
            $result = stripslashes($value);

            if ($forceString || !is_numeric($result)) {
                $search = ['\\', "\0", "\n", "\r", "\x1a", "'", '"'];
                $replace = ['\\\\', '\\0', '\\n', '\\r', "\Z", "\'", '\"'];

                $result = '\'' . strip_tags(nl2br(str_replace($search, $replace, $result))) . '\'';
            }
        } elseif ($forceString) {
            $result = '';
        }

        return $result;
    }

    /**
     * Set API access credentials
     *
     * @param string $host
     * @param string $port
     * @param string $user
     * @param string $password
     *
     * @return $this
     */
    public function setCredentials($host, $port, $user, $password)
    {
        $config = [
            'host'     => $host . ':' . $port,
            'username' => $user,
            'password' => $password,
            'dbname'   => $this->directSqlConfigProvider->getDatabase()
        ];

        $this->connectionConfig = $this->getValidConfig($config);

        return $this;
    }

    /**
     * Get Direct SQL config stored in conf
     *
     * @return array
     */
    protected function getCredentialsFromConfig()
    {
        return [
            'host'     => $this->directSqlConfigProvider->getHost() . ':' . $this->directSqlConfigProvider->getPort(),
            'username' => $this->directSqlConfigProvider->getUser(),
            'password' => $this->directSqlConfigProvider->getPassword(),
            'dbname'   => $this->directSqlConfigProvider->getDatabase()
        ];
    }

    /**
     * Validates the config and adds default options, if any is missing.
     * Duplicated form parent, as parent's method is private
     *
     * @see \Magento\Framework\Model\ResourceModel\Type\Db\Pdo\Mysql::getValidConfig
     *
     * @param array $config
     *
     * @return array
     */
    protected function getValidConfig(array $config)
    {
        $default = ['initStatements' => 'SET NAMES utf8', 'type' => 'pdo_mysql', 'active' => false];
        foreach ($default as $key => $value) {
            if (!isset($config[$key])) {
                $config[$key] = $value;
            }
        }
        $required = ['host'];
        foreach ($required as $name) {
            if (!isset($config[$name])) {
                throw new InvalidArgumentException("MySQL adapter: Missing required configuration option '$name'");
            }
        }

        if (isset($config['port'])) {
            throw new InvalidArgumentException(
                "Port must be configured within host (like '$config[host]:$config[port]') parameter, not within port"
            );
        }

        $config['active'] = !(
            $config['active'] === 'false'
            || $config['active'] === false
            || $config['active'] === '0'
        );

        return $config;
    }
}
