<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-27
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker;

use Exception;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Jobqueue as AbstractJob;
use Fastmag\Sync\Model\Jobqueue\StandardRepository as JobRepository;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\Collection;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Trait Standard
 *
 * Trait for all "standard" workers
 */
trait Standard
{
    /** @var Logger $logger */
    protected $logger;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @var Collection $jobs */
    protected $jobs;

    /**
     * Get worker code
     *
     * @return string
     */
    abstract public function getCode();

    /**
     * Main method
     *
     * @return void
     *
     * @throws ProcessException
     */
    abstract public function run();

    /**
     * Tells if the current worker has some subordinate workers to be run before it
     *
     * @return bool
     */
    public function hasSubordinateWorkersBefore()
    {
        return property_exists($this, 'subordinateWorkersBefore') && count($this->subordinateWorkersBefore) > 0;
    }

    /**
     * Returns the list of workers codes to be run before the current worker
     *
     * @return string[]
     */
    public function getSubordinateWorkersBefore()
    {
        return (property_exists($this, 'subordinateWorkersBefore') ? $this->subordinateWorkersBefore : []);
    }

    /**
     * Tells if the current worker has some subordinate workers to be run after it
     *
     * @return bool
     */
    public function hasSubordinateWorkersAfter()
    {
        return property_exists($this, 'subordinateWorkersAfter') && count($this->subordinateWorkersAfter) > 0;
    }

    /**
     * Returns the list of workers codes to be run after the current worker
     *
     * @return string[]
     */
    public function getSubordinateWorkersAfter()
    {
        return (property_exists($this, 'subordinateWorkersAfter') ? $this->subordinateWorkersAfter : []);
    }

    /**
     * Get jobs list
     *
     * @return Collection $jobs
     */
    public function getJobs()
    {
        return $this->jobs;
    }

    /**
     * Set jobs list
     *
     * @param Collection $jobs
     *
     * @return self
     */
    public function setJobs($jobs)
    {
        $this->jobs = $jobs;

        return $this;
    }

    /**
     * Invalidate job
     *
     * @param AbstractJob $job
     * @param Exception   $exception
     *
     * @return Standard
     */
    public function invalidateJob($job, $exception)
    {
        $this->jobRepository->invalidate($job, $exception);

        $this->error($job, '[ERROR] ' . $exception->getMessage());

        return $this;
    }

    /**
     * Skip job
     *
     * @param AbstractJob $job
     * @param Exception   $exception
     *
     * @return Standard
     */
    public function skipJob($job, $exception)
    {
        $this->jobRepository->skip($job, $exception);

        $this->error($job, '[ERROR] ' . $exception->getMessage());

        return $this;
    }

    /**
     * Save job, or throws an exception
     *
     * @param AbstractJob $job
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function saveJob($job)
    {
        try {
            $this->jobRepository->save($job);
        } catch (CouldNotSaveException $exception) {
            $message = __('[ERROR] Unable to save job.');

            $this->critical($job, $message->render());

            throw new ProcessException($message);
        }
    }

    /**
     * Log debug message
     *
     * @param AbstractJob $job
     * @param string      $message
     *
     * @return void
     */
    protected function debug(AbstractJob $job, string $message): void
    {
        $this->logger->debug(
            '[' . $this->getCode() . '][Job #' . $job->getId() . '][Entity ' . $job->getContentId() . '] '
            . $message
        );
    }

    /**
     * Log info message
     *
     * @param AbstractJob $job
     * @param string      $message
     *
     * @return void
     */
    protected function info(AbstractJob $job, string $message): void
    {
        $this->logger->info(
            '[' . $this->getCode() . '][Job #' . $job->getId() . '][Entity ' . $job->getContentId() . '] '
            . $message
        );
    }

    /**
     * Log notice message
     *
     * @param AbstractJob $job
     * @param string      $message
     *
     * @return void
     */
    protected function notice(AbstractJob $job, string $message): void
    {
        $this->logger->notice(
            '[' . $this->getCode() . '][Job #' . $job->getId() . '][Entity ' . $job->getContentId() . '] '
            . $message
        );
    }

    /**
     * Log warning message
     *
     * @param AbstractJob $job
     * @param string      $message
     *
     * @return void
     */
    protected function warning(AbstractJob $job, string $message): void
    {
        $this->logger->warning(
            '[' . $this->getCode() . '][Job #' . $job->getId() . '][Entity ' . $job->getContentId() . '] '
            . $message
        );
    }

    /**
     * Log error message
     *
     * @param AbstractJob $job
     * @param string      $message
     *
     * @return void
     */
    protected function error(AbstractJob $job, string $message): void
    {
        $this->logger->error(
            '[' . $this->getCode() . '][Job #' . $job->getId() . '][Entity ' . $job->getContentId() . '] '
            . $message
        );
    }

    /**
     * Log critical message
     *
     * @param AbstractJob $job
     * @param string      $message
     *
     * @return void
     */
    protected function critical(AbstractJob $job, string $message): void
    {
        $this->logger->critical(
            '[' . $this->getCode() . '][Job #' . $job->getId() . '][Entity ' . $job->getContentId() . '] '
            . $message
        );
    }

    /**
     * Log alert message
     *
     * @param AbstractJob $job
     * @param string      $message
     *
     * @return void
     */
    protected function alert(AbstractJob $job, string $message): void
    {
        $this->logger->alert(
            '[' . $this->getCode() . '][Job #' . $job->getId() . '][Entity ' . $job->getContentId() . '] '
            . $message
        );
    }

    /**
     * Log emergency message
     *
     * @param AbstractJob $job
     * @param string      $message
     *
     * @return void
     */
    protected function emergency(AbstractJob $job, string $message): void
    {
        $this->logger->emergency(
            '[' . $this->getCode() . '][Job #' . $job->getId() . '][Entity ' . $job->getContentId() . '] '
            . $message
        );
    }
}
