<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-06-19
 ******************************************************************************/

namespace Fastmag\Sync\Model\ResourceModel\Jobqueue;

use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as ModelInterface;
use Fastmag\Sync\Model\Jobqueue\ToMagento as Model;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class ToMagento
 *
 * Resource model for Fastmag to Magento job
 *
 * @method bool delete(ModelInterface $object)
 */
class ToMagento extends AbstractDb
{
    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /**
     * ToMagento constructor.
     *
     * @param Context     $context
     * @param Json        $jsonSerializer
     * @param string|null $connectionName
     */
    public function __construct(Context $context, Json $jsonSerializer, $connectionName = null)
    {
        parent::__construct($context, $connectionName);

        $this->jsonSerializer = $jsonSerializer;
    }

    /**
     * ToMagento constructor
     *
     * @return void
     */
    protected function _construct()
    {
        $this->_init('fastmag_sync_jobqueue_to_magento', 'job_id');
    }

    /**
     * @inheritDoc
     *
     * Decode JSON hydrated data
     */
    public function load(AbstractModel $object, $value, $field = null)
    {
        parent::load($object, $value, $field);

        $dataJson = $object->getData(ModelInterface::HYDRATED_DATA);
        if ($dataJson !== null) {
            $object->setData(ModelInterface::HYDRATED_DATA, $this->jsonSerializer->unserialize($dataJson));
        }

        return $this;
    }

    /**
     * @inheritDoc
     *
     * Encode hydrated data
     */
    public function save(AbstractModel $object)
    {
        $data = $object->getData(ModelInterface::HYDRATED_DATA);
        if (is_array($data)) {
            $object->setData(ModelInterface::HYDRATED_DATA, $this->jsonSerializer->serialize($data));
        }

        parent::save($object);

        return $object->setData(ModelInterface::HYDRATED_DATA, $data);
    }

    /**
     * Load job with unique key (content_id, job_code, priority)
     *
     * @param Model  $job
     * @param int    $contentId
     * @param string $jobCode
     * @param int    $priority
     *
     * @return $this
     *
     * @see AbstractDb::load
     */
    public function loadByUniqueKey($job, $contentId, $jobCode, $priority)
    {
        $connection = $this->getConnection();
        if ($connection && $contentId !== null && $jobCode !== null && $priority !== null) {
            $select = $this->getLoadSelectByUniqueKey($contentId, $jobCode, $priority);

            if ($select !== null) {
                $data = $connection->fetchRow($select);

                if ($data) {
                    $job->setData($data);
                }
            }
        }

        $this->unserializeFields($job);
        $this->_afterLoad($job);
        $job->afterLoad();
        $job->setOrigData();
        $job->setHasDataChanges(false);

        return $this;
    }

    /**
     * Retrieve select job for load object data by unique key (content_id, job_code, priority)
     * Returns null if main table is empty (which won't happen)
     *
     * @param int $contentId
     * @param string $jobCode
     * @param int $priority
     *
     * @return Select|null
     *
     * @see AbstractDb::_getLoadSelect
     */
    protected function getLoadSelectByUniqueKey($contentId, $jobCode, $priority)
    {
        try {
            $mainTable = $this->getMainTable();

            $contentIdField = $this->getConnection()
                ->quoteIdentifier(sprintf('%s.%s', $mainTable, ModelInterface::CONTENT_ID));
            $jobCodeField = $this->getConnection()
                ->quoteIdentifier(sprintf('%s.%s', $mainTable, ModelInterface::JOB_CODE));
            $priorityField = $this->getConnection()
                ->quoteIdentifier(sprintf('%s.%s', $mainTable, ModelInterface::PRIORITY));

            $select = $this->getConnection()->select()
                ->from($mainTable)
                ->where($contentIdField . ' = ?', $contentId)
                ->where($jobCodeField . ' = ?', $jobCode)
                ->where($priorityField . ' = ?', $priority);
        } catch (LocalizedException $exception) {
            $select = null;
        }

        return $select;
    }
}
