<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-06-18
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoBillingAddressException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\ScopeInterface;
use stdClass;

/**
 * Class Data
 *
 * Integration class used for inserting or updating customers from Magento to Fastmag
 */
class Data extends Customer
{
    /** @var string */
    public const DEFAULT_REFERER = 'MAGENTO';

    /** @inheritDoc */
    protected $code = 'tofastmag_integration_customer';

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @var string $hydrationWorker */
    protected $hydrationWorker = 'tofastmag_hydration_customer';

    /** @var string[] $subordinateWorkersAfter */
    protected $subordinateWorkersAfter = ['tofastmag_integration_customer_address'];

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());

            return;
        }

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                $this->currentJob = $job;

                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                } catch (NoBillingAddressException $exception) {
                    $this->skipJob($job, $exception);
                }

                $this->saveJob($job);
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_EXPORT_ENABLE);
    }

    /**
     * @inheritDoc
     *
     * @param Job $job
     *
     * @throws NoBillingAddressException
     */
    protected function processJob($job)
    {
        $entity = $this->getCustomerEntity($job);

        $billingAddress = $entity->getBillingAddress();
        if ($billingAddress === null) {
            throw new NoBillingAddressException(__(
                'Unable to create customer #%1 on Fastmag: no billing address found.',
                $entity->getMagentoId()
            ));
        }

        if ($entity->getFastmagId() === null) {
            $fastmagId = $this->getFastmagIdByEmail($entity->getEmailAddress());

            if ($fastmagId) {
                $entity->setFastmagId($fastmagId);

                $this->saveCustomerSync($entity->getMagentoId(), $fastmagId);
            }
        }

        $newFastmagId = $this->sendDataToFastmag($entity->getMagentoId(), $entity);
        if ($newFastmagId !== false) {
            $entity->setFastmagId($newFastmagId);

            $this->saveCustomerSync($entity->getMagentoId(), $newFastmagId);
        }
    }

    /**
     * Get customer entity from job
     *
     * @param Job $job
     *
     * @return CustomerEntity
     */
    protected function getCustomerEntity($job)
    {
        return $job->getEntity();
    }

    /**
     * Send customer to Fastmag
     *
     * @param int            $customerId
     * @param CustomerEntity $customerEntity
     *
     * @return int|bool new Fastmag ID if it's an insertion, false otherwise
     *
     * @throws JobException
     */
    protected function sendDataToFastmag($customerId, $customerEntity)
    {
        $result = false;

        if ($customerEntity->getFastmagId() === null) {
            $result = $this->createFastmagCustomer($customerId, $customerEntity);
        } else {
            $this->updateFastmagCustomer($customerId, $customerEntity);
        }

        return $result;
    }

    /**
     * Create customer on Fastmag
     *
     * @param int            $magentoId
     * @param CustomerEntity $entity
     *
     * @return int
     *
     * @throws JobException
     */
    protected function createFastmagCustomer($magentoId, $entity)
    {
        $customerData = $this->setDataRequest($entity);

        try {
            $response = $this->api->post('/boa/client/create/index.ips', $customerData);

            if (array_key_exists('data', $response)) {
                $response = $response['data'][0];
            }

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(
                    __($message),
                    $this->api->getLastRequest(),
                    null,
                    (array_key_exists('errorCode', $response) ? $response['errorCode'] : 0)
                );
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to create customer #%1 on Fastmag through API. Code: %2. Message: %3. Cause: %4. Data sent: %5',
                $magentoId,
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                $exception->getRequest()
            ));
        }

        return (int)$response['Client'];
    }

    /**
     * Set the customer entity as intended for the API
     *
     * @param CustomerEntity $entity
     * @param bool           $update
     *
     * @return stdClass
     */
    protected function setDataRequest(CustomerEntity $entity, bool $update = false): stdClass
    {
        /** @var AddressEntity $billingAddress */
        $billingAddress = $entity->getBillingAddress();

        $result = [
            'Nom'           => $entity->getLastname(),
            'Prenom'        => $entity->getFirstname(),
            'Civilite'      => $entity->getPrefix(),
            'Adresse1'      => $billingAddress->getStreetOne(),
            'Adresse2'      => $billingAddress->getStreetTwo(),
            'Adresse3'      => $billingAddress->getStreetThree(),
            'CodePostal'    => $billingAddress->getPostcode(),
            'Ville'         => $billingAddress->getCity(),
            'Pays'          => $billingAddress->getCountryId(),
            'DateNaissance' => $entity->getDob(),
            'Telephone'     => $billingAddress->getPhoneNumber(),
            'Email'         => $entity->getEmailAddress(),
            'Societe'       => $billingAddress->getCompany(),
            'CodeExterne'   => $entity->getMagentoId(),
            'Sexe'          => $entity->getGender(),
            'PubParEMail'   => ($entity->getSubscribeNewsletter() ? '1' : '0')
        ];

        if ($update) {
            $result['Client'] = $entity->getFastmagId();
        } else {
            $result['Magasin'] = $this->config->getValue(
                Config::XML_PATH_ORDER_WORKFLOW_SHOP,
                ScopeInterface::SCOPE_STORES,
                $entity->getStoreId()
            );
            $result['Provenance'] = self::DEFAULT_REFERER;
        }

        foreach ($result as $field => $data) {
            if ($data === null || $data === '') {
                unset($result[$field]);
            }
        }

        return (object)['Clients' => [(object)$result]];
    }

    /**
     * Update customer on Fastmag
     *
     * @param int            $magentoId
     * @param CustomerEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function updateFastmagCustomer($magentoId, $entity)
    {
        $this->checkFastmagEmail($entity);

        $customerData = $this->setDataRequest($entity, true);

        try {
            $response = $this->api->post('/boa/client/update/index.ips', $customerData);

            if (array_key_exists('data', $response)) {
                $response = $response['data'][0];
            }

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(
                    __($message),
                    $this->api->getLastRequest(),
                    null,
                    (array_key_exists('errorCode', $response) ? $response['errorCode'] : 0)
                );
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to update customer #%1 on Fastmag through API. Code: %2. Message: %3. Cause: %4. Data sent: %5',
                $magentoId,
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                $exception->getRequest()
            ));
        }
    }

    /**
     * Check email of the customer on Fastmag, and change it if it's not the same as the one in Magento
     *
     * @param CustomerEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function checkFastmagEmail(CustomerEntity $entity): void
    {
        try {
            $sql = 'SELECT Email AS email
                FROM clients
                WHERE Client = ' . $this->getFastmagSqlConnection()->escape($entity->getFastmagId());

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) > 0 && $rows[0]['email'] !== $entity->getEmailAddress()) {
                $this->changeCustomerEmail($entity->getFastmagId(), $entity->getEmailAddress());
            }
        } catch (Exception $exception) {
            throw new JobException(__(
                'Error when trying to check customer email on Fastmag. Message: %1. Customers IDs: %2',
                $exception->getMessage(),
                $entity->getFastmagId()
            ));
        }
    }

    /**
     * Change customer email on Fastmag
     *
     * @param int    $fastmagCustomerId
     * @param string $newEmail
     *
     * @return void
     *
     * @throws JobException
     */
    protected function changeCustomerEmail(int $fastmagCustomerId, string $newEmail): void
    {
        $data = (object)['Emails' => [(object)['AncienMail' => $fastmagCustomerId, 'NouveauMail' => $newEmail]]];

        try {
            $response = $this->api->post('/boa/client/email/index.ips', $data);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(
                    __($message),
                    $this->api->getLastRequest(),
                    null,
                    (array_key_exists('errorCode', $response) ? $response['errorCode'] : 0)
                );
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to update the customer #%1 email on Fastmag through API. Code: %2. Message: %3. Cause: %4. Data sent: %5',
                $fastmagCustomerId,
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                $exception->getRequest()
            ));
        }
    }
}
