<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2025 HOMEMADE.IO SAS
 * @date      2025-02-28
 ******************************************************************************/

declare(strict_types=1);

namespace Fastmag\Sync\Setup\Patch\Data;

use Exception;
use Fastmag\Sync\Model\Constants;
use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\Product\Action as ProductAction;
use Magento\Catalog\Model\Product\Type;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Eav\Api\AttributeOptionManagementInterface as AttributeOptionManagement;
use Magento\Eav\Model\Config as EavConfig;
use Magento\Eav\Model\Entity\Attribute\AbstractAttribute;
use Magento\Eav\Model\Entity\Attribute\ScopedAttributeInterface;
use Magento\Eav\Setup\EavSetup;
use Magento\Eav\Setup\EavSetupFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Setup\ModuleDataSetupInterface as ModuleDataSetup;
use Magento\Framework\Setup\Patch\DataPatchInterface;
use Magento\Framework\Setup\Patch\PatchRevertableInterface;

/**
 * Class ChangeProductAttributesType
 *
 * Class changing Fastmag-related product attributes to text
 *
 * @SuppressWarnings(PHPMD.LongVariable)
 */
class ChangeProductAttributesType implements DataPatchInterface, PatchRevertableInterface
{
    private ModuleDataSetup $moduleDataSetup;
    private EavConfig $eavConfig;
    private ProductCollectionFactory $productCollectionFactory;
    private ProductAction $productAction;
    private EavSetupFactory $eavSetupFactory;
    private AttributeOptionManagement $attributeOptionManagement;
    private EavSetup $eavSetup;

    private array $attributesToChange = [
        Constants::ATTRIBUTE_PRODUCT_FASTMAG_BRAND_CODE,
        Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUPPLIER_CODE,
        Constants::ATTRIBUTE_PRODUCT_FASTMAG_FAMILY_CODE,
        Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUBFAMILY_CODE,
        Constants::ATTRIBUTE_PRODUCT_FASTMAG_SECTION_CODE,
        Constants::ATTRIBUTE_PRODUCT_FASTMAG_SEASON_CODE,
        Constants::ATTRIBUTE_PRODUCT_FASTMAG_MATERIAL_CODE,
        Constants::ATTRIBUTE_PRODUCT_FASTMAG_MODEL_CODE,
        Constants::ATTRIBUTE_PRODUCT_FASTMAG_THEME_CODE
    ];

    /**
     * ChangeProductAttributesType constructor
     *
     * @param ModuleDataSetup           $moduleDataSetup
     * @param EavConfig                 $eavConfig
     * @param ProductCollectionFactory  $productCollectionFactory
     * @param ProductAction             $productAction
     * @param EavSetupFactory           $eavSetupFactory
     * @param AttributeOptionManagement $attributeOptionManagement
     */
    public function __construct(
        ModuleDataSetup           $moduleDataSetup,
        EavConfig                 $eavConfig,
        ProductCollectionFactory  $productCollectionFactory,
        ProductAction             $productAction,
        EavSetupFactory           $eavSetupFactory,
        AttributeOptionManagement $attributeOptionManagement
    ) {
        $this->moduleDataSetup = $moduleDataSetup;
        $this->eavConfig = $eavConfig;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->productAction = $productAction;
        $this->eavSetupFactory = $eavSetupFactory;
        $this->attributeOptionManagement = $attributeOptionManagement;
    }

    /**
     * @inheritdoc
     */
    public static function getDependencies()
    {
        return [];
    }

    /**
     * @inheritdoc
     */
    public function getAliases()
    {
        return [];
    }

    /**
     * @inheritDoc
     *
     * @throws Exception
     */
    public function apply()
    {
        $this->moduleDataSetup->getConnection()->startSetup();

        $this->eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);
        $sortOrder = 10;

        foreach ($this->attributesToChange as $attributeCode) {
            try {
                $oldAttribute = $this->eavConfig->getAttribute(Product::ENTITY, $attributeCode);
            } catch (LocalizedException $exception) {
                $oldAttribute = null;
            }

            if ($oldAttribute !== null && $oldAttribute->getBackendType() === 'int') {
                $newAttribute = $this->createNewAttribute($oldAttribute, $sortOrder);
                $options = $this->getAttributeOptions($oldAttribute);
                $this->addNewAttributeValuesToCatalog($oldAttribute, $newAttribute, $options);
                $this->replaceOldAttribute($oldAttribute, $newAttribute, $options);
            }

            $sortOrder += 10;
        }

        $this->moduleDataSetup->getConnection()->endSetup();
    }

    /**
     * Create new attribute for saving text values
     *
     * @param AbstractAttribute $attribute
     * @param int               $sortOrder
     *
     * @return AbstractAttribute
     *
     * @throws Exception
     */
    private function createNewAttribute(AbstractAttribute $attribute, int $sortOrder): AbstractAttribute
    {
        $this->eavSetup->addAttribute(
            Product::ENTITY,
            $attribute->getAttributeCode() . '_new',
            [
                'label'        => $attribute->getDefaultFrontendLabel(),
                'type'         => 'varchar',
                'input'        => 'text',
                'user_defined' => true,
                'required'     => false,
                'group'        => Constants::ATTRIBUTE_GROUP_PRODUCT_FASTMAG_CODE,
                'apply_to'     => Type::TYPE_SIMPLE . ',' . Configurable::TYPE_CODE,
                'global'       => ScopedAttributeInterface::SCOPE_GLOBAL,
                'sort_order'   => $sortOrder
            ]
        );

        return $this->eavConfig->getAttribute(Product::ENTITY, $attribute->getAttributeCode() . '_new');
    }

    /**
     * Retrieve all attributes options in an array
     *
     * @param AbstractAttribute $attribute
     *
     * @return array
     */
    private function getAttributeOptions(AbstractAttribute $attribute): array
    {
        $result = [];

        try {
            $options = $attribute->getSource()->getAllOptions(false);
        } catch (LocalizedException $exception) {
            $options = [];
        }

        if (count($options) > 0) {
            foreach ($options as $option) {
                $result[$option['value']] = $option['label'];
            }
        }

        return $result;
    }

    /**
     * Set values for new attributes given
     *
     * @param AbstractAttribute $oldAttribute
     * @param AbstractAttribute $newAttribute
     * @param array             $options
     *
     * @return void
     */
    private function addNewAttributeValuesToCatalog(
        AbstractAttribute $oldAttribute,
        AbstractAttribute $newAttribute,
        array             $options
    ): void {
        if (count($options) > 0) {
            foreach ($options as $value => $label) {
                $collection = $this->productCollectionFactory->create();
                $collection->addAttributeToFilter($oldAttribute->getAttributeCode(), $value);

                if ($collection->count() > 0) {
                    $productIds = $collection->getAllIds();

                    $this->productAction->updateAttributes($productIds, [$newAttribute->getAttributeCode() => $label],
                        0);
                }
            }
        }
    }

    /**
     * Delete old attribute and its options, and renaming new one
     *
     * @param AbstractAttribute $oldAttribute
     * @param AbstractAttribute $newAttribute
     * @param array             $options
     *
     * @return void
     */
    private function replaceOldAttribute(
        AbstractAttribute $oldAttribute,
        AbstractAttribute $newAttribute,
        array             $options
    ): void {
        $oldAttributeCode = $oldAttribute->getAttributeCode();
        $table = $this->moduleDataSetup->getTable('eav_attribute_option_value');

        if (count($options) > 0) {
            foreach ($options as $value => $label) {
                $this->attributeOptionManagement->delete(Product::ENTITY, $oldAttributeCode, $value);

                $this->moduleDataSetup->getConnection()->delete($table, ['option_id = ?' => $value]);
            }
        }

        $this->eavSetup->removeAttribute(Product::ENTITY, $oldAttributeCode);

        $this->eavSetup->updateAttribute(
            Product::ENTITY,
            $newAttribute->getAttributeId(),
            'attribute_code',
            $oldAttributeCode
        );
    }

    /**
     * @inheritDoc
     */
    public function revert()
    {
    }
}
