<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-02-22
 ******************************************************************************/

namespace Fastmag\Sync\Plugin\Jobqueue\ToFastmag;

use Exception;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\Jobqueue\ToFastmagFactory as JobFactory;
use Fastmag\Sync\Plugin\Jobqueue\ToFastmag;
use Fastmag\Sync\Process\Entity\ToFastmag\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Magento\Customer\Api\AddressRepositoryInterface as AddressRepository;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Customer\Api\Data\AddressInterface as AddressModel;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;

/**
 * Class CustomerAddress
 *
 * Plugin on AddressRepository to add Fastmag alias on newly created address
 * and create a matching job in ToFastmag job queue
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class CustomerAddress extends ToFastmag
{
    /** @var CustomerRepository $customerRepository */
    protected $customerRepository;

    /** @var CustomerEntityFactory $customerEntityFactory */
    protected $customerEntityFactory;

    /** @var AddressEntityFactory $addressEntityFactory */
    protected $addressEntityFactory;

    /**
     * AddFastmagAliasToAddress constructor
     *
     * @param JobFactory            $jobFactory
     * @param JobRepository         $jobRepository
     * @param Logger                $logger
     * @param Config                $config
     * @param CustomerRepository    $customerRepository
     * @param AddressEntityFactory  $addressEntityFactory
     * @param CustomerEntityFactory $customerEntityFactory
     */
    public function __construct(
        JobFactory $jobFactory,
        JobRepository $jobRepository,
        Logger $logger,
        Config $config,
        CustomerRepository $customerRepository,
        AddressEntityFactory $addressEntityFactory,
        CustomerEntityFactory $customerEntityFactory
    ) {
        parent::__construct($jobFactory, $jobRepository, $logger, $config);

        $this->customerRepository = $customerRepository;
        $this->addressEntityFactory = $addressEntityFactory;
        $this->customerEntityFactory = $customerEntityFactory;
    }

    /**
     * @see AddressRepositoryInterface::save()
     *
     * @param AddressRepository $subject
     * @param AddressModel      $address
     *
     * @return AddressModel
     */
    public function afterSave(AddressRepository $subject, AddressModel $address)
    {
        if ($address->getId()
            && $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_EXPORT_ENABLE)
            && !$this->isCalledByFastmagModule()
        ) {
            try {
                $customer = $this->customerRepository->getById($address->getCustomerId());
            } catch (Exception $exception) {
                $customer = null;
            }

            if ($customer !== null
                && in_array($address->getId(), [$customer->getDefaultBilling(), $customer->getDefaultShipping()], true)
            ) {
                $job = $this->jobFactory->create();

                if ($address->getId() === $customer->getDefaultBilling()) {
                    $job->setContentId($customer->getId())
                        ->setJobCode('tofastmag_integration_customer');
                } else {
                    $job->setContentId($customer->getId())
                        ->setJobCode('tofastmag_integration_customer_address_shipping');
                }

                try {
                    $this->jobRepository->save($job);
                } catch (Exception $exception) {
                    $this->logger->error($exception->getMessage());
                }
            }
        }

        return $address;
    }

    /**
     * @see AddressRepository::deleteById()
     *
     * @param AddressRepository $subject
     * @param int               $addressId
     *
     * @return void
     */
    public function beforeDeleteById(AddressRepository $subject, int $addressId)
    {
        if ($addressId
            && $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_EXPORT_ENABLE)
            && !$this->isCalledByFastmagModule()
        ) {
            try {
                $address = $subject->getById($addressId);

                $addressAliasAttribute = $address->getCustomAttribute(Constants::ATTRIBUTE_ADDRESS_ALIAS_CODE);
                if ($addressAliasAttribute !== null) {
                    $addressAlias = $addressAliasAttribute->getValue();

                    if ($addressAlias) {
                        $this->addArchivingJob($addressAlias, $address);
                    }
                }
            } catch (LocalizedException $exception) {
                // Do nothing, since it seems that there's no address to archive in Fastmag after all
            }
        }
    }

    /**
     * @see AddressRepository::delete()
     *
     * @param AddressRepository $subject
     * @param AddressModel      $address
     *
     * @return void
     */
    public function beforeDelete(AddressRepository $subject, AddressModel $address)
    {
        if ($this->config->isSetFlag(Config::XML_PATH_CUSTOMER_EXPORT_ENABLE) && !$this->isCalledByFastmagModule()) {
            $addressAliasAttribute = $address->getCustomAttribute(Constants::ATTRIBUTE_ADDRESS_ALIAS_CODE);

            if ($addressAliasAttribute !== null) {
                $addressAlias = $addressAliasAttribute->getValue();

                if ($addressAlias) {
                    $this->addArchivingJob($addressAlias, $address);
                }
            }
        }
    }

    /**
     * Add archiving job in job queue for the deleted address
     *
     * @param string       $alias
     * @param AddressModel $address
     *
     * @return void
     */
    protected function addArchivingJob($alias, $address)
    {
        $job = $this->jobFactory->create();
        try {
            $hydratedData = $this->getDeletedAddressData($alias, $address);

            $job->setContentId($address->getId())
                ->setJobCode('tofastmag_integration_customer_address_archiving')
                ->setHydratedData($hydratedData);

            $this->jobRepository->save($job);
        } catch (CouldNotSaveException $exception) {
            $message = __('Unable to save deleted address #%1 data, archiving job not created', $address->getId());
            $this->logger->error($message);
        } catch (ProcessException $exception) {
            $this->logger->error($exception->getMessage());
        }
    }

    /**
     * Acts as hydration worker to get address data before its deletion
     *
     * @param string       $alias
     * @param AddressModel $address
     *
     * @return array
     *
     * @throws ProcessException
     */
    protected function getDeletedAddressData($alias, $address)
    {
        try {
            $customer = $this->customerRepository->getById($address->getCustomerId());
        } catch (LocalizedException $exception) {
            throw new ProcessException(
                __('Unable to find the customer #%1 of address #%2', $address->getCustomerId(), $address->getId())
            );
        }

        $customerEntity = $this->customerEntityFactory->create();
        $customerEntity->setMagentoId($address->getCustomerId())
            ->setEmailAddress($customer->getEmail());

        $addressEntity = $this->addressEntityFactory->create();

        $addressEntity->setAlias($alias)
            ->setMagentoId($address->getId())
            ->setFirstname($address->getFirstname())
            ->setLastname($address->getLastname())
            ->setCompany($address->getCompany())
            ->setStreet($address->getStreet())
            ->setPostcode($address->getPostcode())
            ->setCity($address->getCity())
            ->setCountryId($address->getCountryId())
            ->setPhoneNumber($address->getTelephone())
            ->setArchive(true)
            ->setIsDefault($address->isDefaultBilling() || $address->isDefaultShipping());

        $customerEntity->setShippingAddress($addressEntity);

        return $customerEntity->export();
    }
}
