<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2025 HOMEMADE.IO SAS
 * @date      2025-02-28
 ******************************************************************************/

declare(strict_types=1);

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Product\Variation as VariationEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToMagento\Integration as IntegrationTrait;
use Magento\Catalog\Api\ProductAttributeOptionManagementInterface as AttributeOptionManager;
use Magento\Catalog\Api\ProductAttributeRepositoryInterface as AttributeRepository;
use Magento\Catalog\Api\Data\ProductAttributeInterface as Attribute;
use Magento\Eav\Api\Data\AttributeOptionInterface as AttributeOption;
use Magento\Eav\Api\Data\AttributeOptionInterfaceFactory as AttributeOptionFactory;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * Class Option
 *
 * Integration worker setting attributes option before saving products
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 * @SuppressWarnings(PHPMD.LongVariable)
 */
class Option extends Worker
{
    use IntegrationTrait;

    /** @inheritDoc */
    protected $code = 'tomagento_integration_product_option';

    private AttributeRepository $attributeRepository;
    private AttributeOptionManager $attributeOptionManager;
    private AttributeOptionFactory $attributeOptionFactory;

    /** @var array $attributesToSync */
    private $attributesToSync;

    /** @var bool $useColorDescrption */
    private $useColorDescrption;

    /**
     * Option constructor
     *
     * @param Logger                 $logger
     * @param JobRepository          $jobRepository
     * @param Config                 $config
     * @param AttributeRepository    $attributeRepository
     * @param AttributeOptionManager $attributeOptionManager
     * @param AttributeOptionFactory $attributeOptionFactory
     */
    public function __construct(
        Logger                 $logger,
        JobRepository          $jobRepository,
        Config                 $config,
        AttributeRepository    $attributeRepository,
        AttributeOptionManager $attributeOptionManager,
        AttributeOptionFactory $attributeOptionFactory
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->attributeRepository = $attributeRepository;
        $this->attributeOptionManager = $attributeOptionManager;
        $this->attributeOptionFactory = $attributeOptionFactory;

        $this->useColorDescrption =
            $this->config->isSetFlag(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_USE_COLOR_DESCRIPTION);
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());

            return;
        }

        $this->setAttributesToSync();

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                }

                $this->saveJob($job);
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ENABLE);
    }

    /**
     * Set the attributes list to sync
     *
     * @return void
     */
    private function setAttributesToSync(): void
    {
        $result = [];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_TEXTILE)) {
            $result['washing'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_WASHING_CODE;
            $result['chlorine'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_CHLORINE_CODE;
            $result['ironing'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_IRONING_CODE;
            $result['dry_cleaning'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRY_CLEANING_CODE;
            $result['drying'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRYING_CODE;
            $result['water'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_WATER_CODE;
        }

        $this->attributesToSync = $result;
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var ProductEntity $entity */
        $entity = $job->getEntity();

        if ($this->productMustBeSynced($entity)) {
            if (count($this->attributesToSync) > 0) {
                foreach ($this->attributesToSync as $fieldCode => $attributeCode) {
                    $label = $entity->getData($fieldCode);

                    $this->saveOption($attributeCode, $label);
                }
            }

            foreach ($entity->getVariationsFlat() as $variation) {
                $this->saveColorSizeOptions($variation);
            }
        }
    }

    /**
     * Check if products fill all the condition to be synced
     *
     * @param ProductEntity $entity
     *
     * @return bool
     */
    private function productMustBeSynced($entity): bool
    {
        $result = ($entity->getActive() && $entity->getVisibleWeb())
            || !$this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ACTIVE_VISIBLEWEB);

        $onlyDefinedStocks = $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS);
        if ($onlyDefinedStocks && $entity->getStockLevel() === 0) {
            $result = false;
        }

        if ($entity->getMagentoId()) {
            $result = true;
        }

        return $result;
    }

    /**
     * Create or update option value with all the i18nized labels
     *
     * @param int|string $attributeCode
     * @param string     $label
     *
     * @return void
     *
     * @throws JobException
     */
    private function saveOption($attributeCode, $label): void
    {
        try {
            $attribute = $this->attributeRepository->get($attributeCode);
        } catch (NoSuchEntityException $exception) {
            throw new JobException(__('Attribute "%1" does not exist in Magento', $attributeCode));
        }

        if (!empty($label)) {
            $option = $this->getOption($attribute, $label);

            if (!$option) {
                $option = $this->attributeOptionFactory->create();
                $option->setLabel($label);

                try {
                    $this->attributeOptionManager->add($attributeCode, $option);
                } catch (Exception $exception) {
                    throw new JobException(__(
                        'Exception when adding option "%1" of the attribute "%2": %3',
                        $label,
                        $attributeCode,
                        $exception->getMessage()
                    ));
                }
            }
        }
    }

    /**
     * Check if option already exists for attribute
     *
     * @param Attribute $attribute
     * @param string    $value
     *
     * @return AttributeOption|false
     */
    private function getOption($attribute, $value)
    {
        $result = false;

        if ($attribute->usesSource()) {
            $options = $attribute->getOptions();

            foreach ($options as $option) {
                if (strcasecmp($option->getLabel(), trim($value)) === 0) {
                    $result = $option;
                    break;
                }
            }
        }

        return $result;
    }

    /**
     * Save color and size options given the config of a product's variation
     *
     * @param VariationEntity $entity
     *
     * @returns void
     *
     * @throws JobException
     */
    private function saveColorSizeOptions($entity): void
    {
        $colorAttributeId = $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_COLOR);

        if ($this->useColorDescrption) {
            $this->saveOption($colorAttributeId, $entity->getData('color_description'));

            $this->saveOption(
                $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_INTERNAL_COLOR),
                $entity->getData('color')
            );
        } else {
            $this->saveOption($colorAttributeId, $entity->getData('color'));
        }

        $this->saveOption(
            $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_SIZE),
            $entity->getData('size')
        );
    }
}
