<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-01-02
 ******************************************************************************/

namespace Fastmag\Sync\Controller\Adminhtml\Catalog\Product;

use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterfaceFactory as JobFactory;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\System\Connection\Api as FastmagApi;
use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Catalog\Api\Data\ProductInterface as ProductModel;
use Magento\Catalog\Api\ProductRepositoryInterface as ProductRepository;
use Magento\Catalog\Model\Product\Type;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use stdClass;

/**
 * Class Resync
 *
 * Controller class creating an import job to sync product in Magento
 */
class Resync extends Action
{
    /** @var string */
    public const ADMIN_RESOURCE = 'Fastmag_Sync::product_resync';

    protected JobRepository $jobRepository;
    protected JobFactory $jobFactory;
    protected ProductRepository $productRepository;
    protected FastmagApi $api;

    /** @var ProductModel $currentProduct */
    protected $currentProduct;

    /**
     * Resync constructor
     *
     * @param Context           $context
     * @param JobRepository     $jobRepository
     * @param JobFactory        $jobFactory
     * @param ProductRepository $productRepository
     * @param FastmagApi        $api
     */
    public function __construct(
        Context           $context,
        JobRepository     $jobRepository,
        JobFactory        $jobFactory,
        ProductRepository $productRepository,
        FastmagApi        $api
    ) {
        parent::__construct($context);

        $this->jobRepository = $jobRepository;
        $this->jobFactory = $jobFactory;
        $this->productRepository = $productRepository;
        $this->api = $api;
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        $productId = $this->getRequest()->getParam('id');

        try {
            $this->currentProduct = $this->productRepository->getById($productId);
        } catch (NoSuchEntityException $exception) {
            $this->messageManager->addErrorMessage(
                __('Unable to get product #%1', $productId)->render()
            );
        }

        $fastmagRef = $this->getFastmagRef();

        if ($fastmagRef !== null) {
            $job = $this->jobFactory->create();

            $job->setContentId($fastmagRef)
                ->setJobCode('tomagento_integration_product_update');

            try {
                $this->jobRepository->save($job);

                $this->messageManager->addSuccessMessage(
                    __('Product "%1" will be resynchronized in a few moments', $fastmagRef)->render()
                );
            } catch (CouldNotSaveException $exception) {
                $this->messageManager->addErrorMessage(
                    __('Unable to create a sync job for the product: %1', $exception->getMessage())->render()
                );
            }
        } else {
            $this->messageManager->addErrorMessage(
                __('Product #%1 does not seem to exist in Fastmag, it does not have a Fastmag ID', $productId)->render()
            );
        }

        $resultRedirect = $this->resultRedirectFactory->create();
        $resultRedirect->setPath('catalog/product/edit', ['id' => $productId]);

        return $resultRedirect;
    }

    /**
     * Get product Fastmag reference
     *
     * @return string|null
     */
    protected function getFastmagRef()
    {
        $result = null;

        if ($this->currentProduct) {
            if ($this->currentProduct->getTypeId() === Configurable::TYPE_CODE) {
                $result = $this->currentProduct->getSku();
            } elseif (in_array($this->currentProduct->getTypeId(), [Type::TYPE_SIMPLE, Type::TYPE_VIRTUAL], true)) {
                $fastmagProductId = $this->getProductFastmagId();

                if ($fastmagProductId !== null) {
                    $result = $this->getProductRefByFastmagId($fastmagProductId);
                }
            }
        }

        return $result;
    }

    /**
     * Get product Fastmag ID
     *
     * @return int|null
     */
    protected function getProductFastmagId(): ?int
    {
        $result = null;

        $fastmagProductIdAttribute
                = $this->currentProduct->getCustomAttribute(Constants::ATTRIBUTE_PRODUCT_FASTMAG_PRODUCT_ID_CODE);

        if ($fastmagProductIdAttribute !== null) {
            $result = $fastmagProductIdAttribute->getValue();
        }

        return $result;
    }

    /**
     * Request Fastmag API to get product ref
     *
     * @param int $fastmagProductId
     *
     * @return string|null
     */
    protected function getProductRefByFastmagId(int $fastmagProductId): ?string
    {
        $result = null;

        $request = $this->generateProductRequest($fastmagProductId);

        try {
            $response = $this->api->post('/boa/product/search/index.ips', $request);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(
                    __($message),
                    $this->api->getLastRequest(),
                    null,
                    (array_key_exists('errorCode', $response) ? $response['errorCode'] : 0)
                );
            }

            if (array_key_exists('data', $response)) {
                $response = $response['data'][0];

                $result = $response['Reference'];
            }
        } catch (ApiException $exception) {
            // Do nothing
        }

        return $result;
    }

    /**
     * Generate product search request
     *
     * @param int $fastmagGencod
     *
     * @return stdClass
     */
    protected function generateProductRequest(int $fastmagGencod): stdClass
    {
        return (object)['Reference' => '#' . $fastmagGencod];
    }
}
