<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2025 HOMEMADE.IO SAS
 * @date      2025-04-18
 ******************************************************************************/

namespace Fastmag\Sync\Model\Api;

use Fastmag\Sync\Api\ConnectionRepositoryInterface;
use Fastmag\Sync\Api\Data\Connection\StatusInterfaceFactory as StatusFactory;
use Fastmag\Sync\Api\Data\Connection\StatusResponseInterface as StatusResponse;
use Fastmag\Sync\Api\Data\Connection\StatusResponseInterfaceFactory as StatusResponseFactory;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Model\System\Connection\Api as ApiConnector;
use Fastmag\Sync\Model\System\Connection\Sql\DirectSql as MysqlConnector;
use Fastmag\Sync\Model\System\Connection\Sql\EdiSql as EdiQueryConnector;

/**
 * Class ConnectionRepository
 *
 * Implements actual code of Fastmag connection status check REST API
 *
 * @SuppressWarnings(PHPMD.LongVariable)
 * phpcs:disable Magento2.Commenting.ClassPropertyPHPDocFormatting.Missing
 */
class ConnectionRepository implements ConnectionRepositoryInterface
{
    private ApiConnector $apiConnector;
    private EdiQueryConnector $ediQueryConnector;
    private MysqlConnector $mysqlConnector;
    private StatusResponseFactory $statusResponseFactory;
    private StatusFactory $statusFactory;

    /**
     * ConnectionRepository constructor
     *
     * @param ApiConnector          $apiConnector
     * @param EdiQueryConnector     $ediQueryConnector
     * @param MysqlConnector        $mysqlConnector
     * @param StatusResponseFactory $statusResponseFactory
     * @param StatusFactory         $statusFactory
     */
    public function __construct(
        ApiConnector          $apiConnector,
        EdiQueryConnector     $ediQueryConnector,
        MysqlConnector        $mysqlConnector,
        StatusResponseFactory $statusResponseFactory,
        StatusFactory         $statusFactory
    ) {
        $this->apiConnector = $apiConnector;
        $this->ediQueryConnector = $ediQueryConnector;
        $this->mysqlConnector = $mysqlConnector;
        $this->statusResponseFactory = $statusResponseFactory;
        $this->statusFactory = $statusFactory;
    }

    /**
     * @inheritDoc
     */
    public function checkStatus(): StatusResponse
    {
        $response = $this->statusResponseFactory->create();

        $apiResult = $this->statusFactory->create();
        try {
            $this->apiConnector->authenticate();
            $apiResult->setStatus(true);
        } catch (ApiException $exception) {
            $apiResult->setStatus(false)
                ->setMessage(__(
                    'API connection error: Code: %1 - Error: %2 - Cause: %3',
                    $exception->getCode(),
                    $exception->getMessage(),
                    ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : '')
                ));
        }
        $response->setApi($apiResult);

        $ediResult = $this->statusFactory->create();
        try {
            $this->ediQueryConnector->connect();
            $ediResult->setStatus(true);
        } catch (NoConnectionException $exception) {
            $ediResult->setStatus(false)
                ->setMessage(__('EDI connection error: %1', $exception->getMessage()));
        }
        $response->setEdi($ediResult);

        $mysqlResult = $this->statusFactory->create();
        try {
            $this->mysqlConnector->connect();
            $mysqlResult->setStatus(true);
        } catch (NoConnectionException $exception) {
            $mysqlResult->setStatus(false)
                ->setMessage(__('Proxy connection error: %1', $exception->getMessage()));
        }
        $response->setMysql($mysqlResult);

        return $response;
    }
}
