<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2025 HOMEMADE.IO SAS
 * @date      2025-09-03
 ******************************************************************************/

namespace Fastmag\Sync\Model;

use Exception;
use Fastmag\Sync\Api\CustomerRepositoryInterface;
use Fastmag\Sync\Api\Data\CustomerInterface as Customer;
use Fastmag\Sync\Api\Data\CustomerSearchResultsInterfaceFactory as SearchResultsFactory;
use Fastmag\Sync\Model\ResourceModel\Customer as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Customer\CollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface as CollectionProcessor;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * Class CustomerRepository
 *
 * Repository of synchronized customers model
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 * @SuppressWarnings(PHPMD.LongVariable)
 * phpcs:disable Magento2.Commenting.ClassPropertyPHPDocFormatting.Missing
 */
class CustomerRepository implements CustomerRepositoryInterface
{
    private ResourceModel $resource;
    private CustomerFactory $customerFactory;
    private CollectionFactory $collectionFactory;
    private SearchResultsFactory $searchResultsFactory;

    /** @var CollectionProcessor $collectionProcessor */
    private $collectionProcessor;

    /**
     * CustomerRepository constructor
     *
     * @param ResourceModel            $resource
     * @param CustomerFactory          $customerFactory
     * @param CollectionFactory        $collectionFactory
     * @param SearchResultsFactory     $searchResultsFactory
     * @param CollectionProcessor|null $collectionProcessor
     */
    public function __construct(
        ResourceModel        $resource,
        CustomerFactory      $customerFactory,
        CollectionFactory    $collectionFactory,
        SearchResultsFactory $searchResultsFactory,
        ?CollectionProcessor $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->customerFactory = $customerFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(Customer $customer): Customer
    {
        try {
            $this->resource->save($customer);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $customer;
    }

    /**
     * @inheritDoc
     */
    public function getById($entityId): Customer
    {
        $customer = $this->customerFactory->create();
        $this->resource->load($customer, $entityId);
        if (!$customer->getId()) {
            throw new NoSuchEntityException(__('The customer with the "%1" ID doesn\'t exist.', $entityId));
        }

        return $customer;
    }

    /**
     * @inheritDoc
     */
    public function getByMagentoId($magentoId)
    {
        $customer = $this->customerFactory->create();
        $this->resource->loadByMagentoId($customer, $magentoId);
        if (!$customer->getId()) {
            throw new NoSuchEntityException(__('The customer with the Magento ID "%1" doesn\'t exist.', $magentoId));
        }

        return $customer;
    }

    /**
     * @inheritDoc
     */
    public function getByFastmagId($fastmagId)
    {
        $customer = $this->customerFactory->create();
        $this->resource->loadByFastmagId($customer, $fastmagId);
        if (!$customer->getId()) {
            throw new NoSuchEntityException(__('The customer with the Fastmag ID "%1" doesn\'t exist.', $fastmagId));
        }

        return $customer;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(Customer $customer): bool
    {
        try {
            $this->resource->delete($customer);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($entityId)
    {
        return $this->delete($this->getById($entityId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessor
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
