<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2025 HOMEMADE.IO SAS
 * @date      2025-09-03
 ******************************************************************************/

namespace Fastmag\Sync\Model;

use Exception;
use Fastmag\Sync\Api\Data\OrderInterface as Order;
use Fastmag\Sync\Api\Data\OrderSearchResultsInterfaceFactory as SearchResultsFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as Ordertransaction;
use Fastmag\Sync\Api\OrderRepositoryInterface;
use Fastmag\Sync\Model\ResourceModel\Order as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Order\CollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface as CollectionProcessor;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * Class OrderRepository
 *
 * Repository class for synchronized orders model
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 * @SuppressWarnings(PHPMD.LongVariable)
 * phpcs:disable Magento2.Commenting.ClassPropertyPHPDocFormatting.Missing
 */
class OrderRepository implements OrderRepositoryInterface
{
    private ResourceModel $resource;
    private OrderFactory $orderFactory;
    private CollectionFactory $collectionFactory;
    private SearchResultsFactory $searchResultsFactory;
    private SearchCriteriaBuilder $searchCriteriaBuilder;
    private SortOrderBuilder $sortOrderBuilder;

    /** @var CollectionProcessor $collectionProcessor */
    private $collectionProcessor;

    /**
     * OrderRepository constructor
     *
     * @param ResourceModel            $resource
     * @param OrderFactory             $orderFactory
     * @param CollectionFactory        $collectionFactory
     * @param SearchResultsFactory     $searchResultsFactory
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param SortOrderBuilder         $sortOrderBuilder
     * @param CollectionProcessor|null $collectionProcessor
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        ResourceModel         $resource,
        OrderFactory          $orderFactory,
        CollectionFactory     $collectionFactory,
        SearchResultsFactory  $searchResultsFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder      $sortOrderBuilder,
        ?CollectionProcessor  $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->orderFactory = $orderFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(Order $order): Order
    {
        try {
            $this->resource->save($order);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $order;
    }

    /**
     * @inheritDoc
     */
    public function getById($syncId): Order
    {
        $order = $this->orderFactory->create();
        $this->resource->load($order, $syncId);
        if (!$order->getId()) {
            throw new NoSuchEntityException(__('The order synchronization with #%1 ID doesn\'t exist.', $syncId));
        }

        return $order;
    }

    /**
     * @inheritDoc
     */
    public function getLastSyncByOrderId(
        $orderId,
        $excludeTypes = [Ordertransaction::FASTMAG_TRANSACTION_TYPE_TRANSFER]
    ) {
        $sortOrderResultAt = $this->sortOrderBuilder->setField(Order::RESULT_AT)
            ->setDescendingDirection()
            ->create();

        $sortOrderEntityId = $this->sortOrderBuilder->setField(Order::SYNC_ID)
            ->setDescendingDirection()
            ->create();

        $this->searchCriteriaBuilder->addFilter(Order::ORDER_ID, $orderId)
            ->addSortOrder($sortOrderResultAt)
            ->addSortOrder($sortOrderEntityId);

        if (count($excludeTypes) > 0) {
            $this->searchCriteriaBuilder->addFilter(Order::TRANSACTION_TYPE, $excludeTypes, 'nin');
        }

        $criteria = $this->searchCriteriaBuilder->create();

        $syncList = $this->getList($criteria);

        if ($syncList->getTotalCount() === 0) {
            throw new NoSuchEntityException(__('There\'s no line on synced orders table for the order #%1', $orderId));
        }

        $items = $syncList->getItems();

        return array_shift($items);
    }

    /**
     * @inheritDoc
     */
    public function getLastSyncByTransactionId($transactionId)
    {
        $sortOrderResultAt = $this->sortOrderBuilder->setField(Order::RESULT_AT)
            ->setDescendingDirection()
            ->create();

        $sortOrderEntityId = $this->sortOrderBuilder->setField(Order::SYNC_ID)
            ->setDescendingDirection()
            ->create();

        $criteria = $this->searchCriteriaBuilder->addFilter(Order::TRANSACTION_ID, $transactionId)
            ->addSortOrder($sortOrderResultAt)
            ->addSortOrder($sortOrderEntityId)
            ->create();

        $syncList = $this->getList($criteria);

        if ($syncList->getTotalCount() === 0) {
            throw new NoSuchEntityException(
                __('There\'s no line on synced orders table for the transaction #%1', $transactionId)
            );
        }

        $items = $syncList->getItems();

        return array_shift($items);
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function getListByOrderId($orderId)
    {
        $criteria = $this->searchCriteriaBuilder->addFilter(Order::ORDER_ID, $orderId)
            ->create();

        return $this->getList($criteria);
    }

    /**
     * @inheritDoc
     */
    public function delete(Order $order): bool
    {
        try {
            $this->resource->delete($order);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($entityId)
    {
        return $this->delete($this->getById($entityId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessor
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
