<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2025 HOMEMADE.IO SAS
 * @date      2025-09-03
 ******************************************************************************/

namespace Fastmag\Sync\Model\Jobqueue;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoScheduledInterface as Job;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoScheduledInterfaceFactory as JobFactory;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoScheduledSearchResultsInterfaceFactory as SearchResultsFactory;
use Fastmag\Sync\Api\Jobqueue\ToMagentoScheduledRepositoryInterface;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagentoScheduled as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagentoScheduled\CollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface as CollectionProcessor;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * Class ToMagentoScheduledRepository
 *
 * Repository of jobs used for scheduled Fastmag to Magento synchronization
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 * @SuppressWarnings(PHPMD.LongVariable)
 * phpcs:disable Magento2.Commenting.ClassPropertyPHPDocFormatting.Missing
 */
class ToMagentoScheduledRepository implements ToMagentoScheduledRepositoryInterface
{
    private ResourceModel $resource;
    private JobFactory $jobFactory;
    private CollectionFactory $collectionFactory;
    private SearchResultsFactory $searchResultsFactory;

    /** @var CollectionProcessor $collectionProcessor */
    private $collectionProcessor;

    /**
     * ToMagentoScheduledRepository constructor
     *
     * @param ResourceModel            $resource
     * @param JobFactory               $jobFactory
     * @param CollectionFactory        $collectionFactory
     * @param SearchResultsFactory     $searchResultsFactory
     * @param CollectionProcessor|null $collectionProcessor
     */
    public function __construct(
        ResourceModel        $resource,
        JobFactory           $jobFactory,
        CollectionFactory    $collectionFactory,
        SearchResultsFactory $searchResultsFactory,
        ?CollectionProcessor $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->jobFactory = $jobFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(Job $job)
    {
        try {
            $this->resource->save($job);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getById($jobId)
    {
        $job = $this->jobFactory->create();
        $this->resource->load($job, $jobId);
        if (!$job->getId()) {
            throw new NoSuchEntityException(__('The job with the "%1" ID doesn\'t exist.', $jobId));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(Job $job)
    {
        try {
            $this->resource->delete($job);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($jobId)
    {
        return $this->delete($this->getById($jobId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessor
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
