<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2026 HOMEMADE.IO SAS
 * @date      2026-01-15
 ******************************************************************************/

declare(strict_types=1);

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Order;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Entity\ToMagento\Order\Creditmemo\ItemFactory as ItemEntityFactory;
use Fastmag\Sync\Process\Entity\ToMagento\Order\CreditmemoFactory as CreditmemoEntityFactory;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Creditmemo
 *
 * Hydration class used for inserting and updating creditmemos from Fastmag to Magento
 *
 * @SuppressWarnings(PHPMD.LongVariable)
 * phpcs:disable Magento2.Commenting.ClassPropertyPHPDocFormatting.Missing
 */
class Creditmemo extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_order_creditmemo';

    protected CreditmemoEntityFactory $creditmemoEntityFactory;
    protected ItemEntityFactory $itemEntityFactory;

    /**
     * Creditmemo constructor
     *
     * @param Logger                  $logger
     * @param JobRepository           $jobRepository
     * @param Config                  $config
     * @param Json                    $jsonSerializer
     * @param Proxy                   $fastmagSql
     * @param CreditmemoEntityFactory $creditmemoEntityFactory
     * @param ItemEntityFactory       $itemEntityFactory
     */
    public function __construct(
        Logger                  $logger,
        JobRepository           $jobRepository,
        Config                  $config,
        Json                    $jsonSerializer,
        Proxy                   $fastmagSql,
        CreditmemoEntityFactory $creditmemoEntityFactory,
        ItemEntityFactory       $itemEntityFactory
    ) {
        parent::__construct($logger, $jobRepository, $config, $jsonSerializer, $fastmagSql);

        $this->creditmemoEntityFactory = $creditmemoEntityFactory;
        $this->itemEntityFactory = $itemEntityFactory;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $results = $this->getDataFromFastmag();
        } catch (ProcessException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        if (isset($results)) {
            foreach ($this->getJobs()->getItems() as $job) {
                if (array_key_exists($job->getContentId(), $results)) {
                    $this->hydrateJob($job, $results[$job->getContentId()]);
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $ordersIds = $this->getJobs()->getColumnValues(Job::CONTENT_ID);

        try {
            $sql = 'SELECT v.Vente AS fastmag_creditmemo_id,
                       v.Nature AS type,
                       v.Total AS grand_total,
                       v.Quantite AS total_quantity,
                       l.Ligne AS fastmag_line_id,
                       l.BarCode AS ref,
                       l.Taille AS size,
                       l.Couleur AS color,
                       l.Quantite AS quantity,
                       l.Total AS total,
                       vo.Vente AS fastmag_order_id
                FROM ventes AS v LEFT JOIN lignes AS l ON v.Vente = l.Vente
                    LEFT JOIN lignescommentaire AS lc ON l.Ligne = lc.IDLigne
                    LEFT JOIN lignes AS lo ON lc.OrigineRetour = lo.Ligne
                    LEFT JOIN ventes AS vo ON lo.Vente = vo.Vente
                WHERE v.Vente IN (' . $this->getFastmagSqlConnection()->escape($ordersIds) . ')
                ORDER BY v.Vente, l.Ligne';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new ProcessException(__(
                'Error when hydrating orders. Message: %1. Orders IDs: %2',
                $exception->getMessage(),
                implode(', ', $ordersIds)
            ));
        }

        $result = [];
        foreach ($rows as $row) {
            $orderId = $row['fastmag_creditmemo_id'];

            if (!array_key_exists($orderId, $result)) {
                if ($row['type'] !== 'AVOIR') {
                    $this->logger->debug(
                        __(
                            '[%1] Order #%2 is not a credit memo, skipping job',
                            $this->getCode(),
                            $row['fastmag_creditmemo_id']
                        )->render()
                    );

                    continue;
                }

                if ($row['fastmag_order_id'] === null) {
                    $this->logger->debug(
                        __(
                            '[%1] Credit memo #%2 not linked to an order, it will not be synced with Magento',
                            $this->getCode(),
                            $row['fastmag_creditmemo_id']
                        )->render()
                    );
                } else {
                    $this->logger->debug(
                        __(
                            '[%1] Getting data for credit memo #%2',
                            $this->getCode(),
                            $row['fastmag_creditmemo_id']
                        )->render()
                    );

                    $creditmemoEntity = $this->creditmemoEntityFactory->create();
                    $creditmemoEntity->setFastmagId((int)$row['fastmag_creditmemo_id'])
                        ->setGrandTotal(-$row['grand_total'])
                        ->setTotalQuantity(-$row['total_quantity']);

                    $result[$orderId] = $creditmemoEntity;
                }
            }

            $itemEntity = $this->itemEntityFactory->create();
            $itemEntity->setFastmagId((int)$row['fastmag_line_id'])
                ->setRef($row['ref'])
                ->setSize($row['size'])
                ->setColor($row['color'])
                ->setQuantity(-$row['quantity'])
                ->setTotal(-$row['total']);
            $result[$orderId]->addItem($itemEntity);

            if ($result[$orderId]->getOrderFastmagId() === null) {
                $result[$orderId]->setOrderFastmagId((int)$row['fastmag_order_id']);
            } elseif ($result[$orderId]->getOrderFastmagId() !== (int)$row['fastmag_order_id']) {
                throw new ProcessException(
                    __(
                        '[%1] There is different orders linked to the same credit memo: %2 and %3',
                        $this->getCode(),
                        $orderId,
                        $row['fastmag_order_id']
                    )
                );
            }
        }

        return $result;
    }
}
