<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2026 HOMEMADE.IO SAS
 * @date      2026-01-16
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Order;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Entity\ToMagento\OrderFactory as OrderEntityFactory;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Status
 *
 * Hydration class used for inserting and updating orders from Fastmag to Magento
 */
class Status extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_order_status';

    /** @var string[] $subordinateWorkersAfter */
    protected $subordinateWorkersAfter = ['tomagento_hydration_order_creditmemo'];

    /** @var OrderEntityFactory $orderEntityFactory */
    protected $orderEntityFactory;

    /**
     * Integration constructor.
     *
     * @param Logger             $logger
     * @param JobRepository      $jobRepository
     * @param Config             $config
     * @param Json               $jsonSerializer
     * @param Proxy              $fastmagSql
     * @param OrderEntityFactory $orderEntityFactory
     */
    public function __construct(
        Logger             $logger,
        JobRepository      $jobRepository,
        Config             $config,
        Json               $jsonSerializer,
        Proxy              $fastmagSql,
        OrderEntityFactory $orderEntityFactory
    ) {
        parent::__construct($logger, $jobRepository, $config, $jsonSerializer, $fastmagSql);

        $this->orderEntityFactory = $orderEntityFactory;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $results = $this->getDataFromFastmag();
        } catch (ProcessException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        if (isset($results)) {
            foreach ($this->getJobs()->getItems() as $job) {
                if (array_key_exists($job->getContentId(), $results)) {
                    $this->hydrateJob($job, $results[$job->getContentId()]);
                } else {
                    $exception = new JobException(__('Order not found in Fastmag'));
                    $this->skipJob($job, $exception);
                }

                try {
                    $this->saveJob($job);
                } catch (ProcessException $exception) {
                    $this->invalidateJob($job, $exception);
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $ordersIds = $this->getJobs()->getColumnValues(Job::CONTENT_ID);

        try {
            $sql = 'SELECT v.Vente AS fastmag_id,
                    v.VShopStatus AS status,
                    IF(LENGTH(v.ListeColis) = 0, NULL, v.ListeColis) AS tracking_number,
                    ve.Transporteur AS carrier
                FROM ventes AS v LEFT OUTER JOIN ventes_etats AS ve ON (v.Vente = ve.Vente)
                WHERE v.Vente IN (' . $this->getFastmagSqlConnection()->escape($ordersIds) . ')';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new ProcessException(__(
                'Error when hydrating orders. Message: %1. Orders IDs: %2',
                $exception->getMessage(),
                implode(', ', $ordersIds)
            ));
        }

        $result = [];
        foreach ($rows as $row) {
            $orderEntity = $this->orderEntityFactory->create();
            $orderEntity->setData($row);

            $result[$row['fastmag_id']] = $orderEntity;
        }

        return $result;
    }
}
