<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2026 HOMEMADE.IO SAS
 * @date      2026-01-16
 ******************************************************************************/

declare(strict_types=1);

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Order;

use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Data\OrderInterface as OrderSync;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\OrderRepositoryInterface as OrderSyncRepository;
use Fastmag\Sync\Api\Service\GenerateSimpleProductSkuInterface as SimpleProductSkuGenerator;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Exception\TransactionHistoryException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Process\Entity\ToMagento\Order\Creditmemo as CreditmemoEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Order\Creditmemo\Item as ItemEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToMagento\Integration as IntegrationTrait;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Sales\Api\Data\CreditmemoCreationArgumentsInterfaceFactory as CreditmemoArgumentsFactory;
use Magento\Sales\Api\Data\CreditmemoItemCreationInterfaceFactory as CreditmemoItemFactory;
use Magento\Sales\Api\Data\OrderInterface as Order;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;
use Magento\Sales\Api\RefundOrderInterface as RefundOrderService;

/**
 * Class Status
 *
 * Integration worker for tomagento_integration_order_status jobs
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 * @SuppressWarnings(PHPMD.LongVariable)
 * phpcs:disable Magento2.Commenting.ClassPropertyPHPDocFormatting.Missing
 */
class Creditmemo extends Worker
{
    use IntegrationTrait;

    /** @inheritDoc */
    protected $code = 'tomagento_integration_order_creditmemo';

    protected OrderSyncRepository $orderSyncRepository;
    protected OrderRepository $orderRepository;
    protected RefundOrderService $refundOrderService;
    protected CreditmemoItemFactory $creditmemoItemFactory;
    protected CreditmemoArgumentsFactory $creditmemoArgumentsFactory;
    protected SimpleProductSkuGenerator $simpleProductSkuGenerator;

    /** @var Job $currentJob */
    protected $currentJob;

    /**
     * Creditmemo constructor
     *
     * @param Logger                     $logger
     * @param JobRepository              $jobRepository
     * @param Config                     $config
     * @param OrderSyncRepository        $orderSyncRepository
     * @param OrderRepository            $orderRepository
     * @param RefundOrderService         $refundOrderService
     * @param CreditmemoItemFactory      $creditmemoItemFactory
     * @param CreditmemoArgumentsFactory $creditmemoArgumentsFactory
     * @param SimpleProductSkuGenerator  $simpleProductSkuGenerator
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        Logger                     $logger,
        JobRepository              $jobRepository,
        Config                     $config,
        OrderSyncRepository        $orderSyncRepository,
        OrderRepository            $orderRepository,
        RefundOrderService         $refundOrderService,
        CreditmemoItemFactory      $creditmemoItemFactory,
        CreditmemoArgumentsFactory $creditmemoArgumentsFactory,
        SimpleProductSkuGenerator  $simpleProductSkuGenerator,
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->orderSyncRepository = $orderSyncRepository;
        $this->orderRepository = $orderRepository;
        $this->refundOrderService = $refundOrderService;
        $this->creditmemoItemFactory = $creditmemoItemFactory;
        $this->creditmemoArgumentsFactory = $creditmemoArgumentsFactory;
        $this->simpleProductSkuGenerator = $simpleProductSkuGenerator;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());

            return;
        }

        foreach ($this->getJobs()->getItems() as $job) {
            $this->currentJob = $job;

            if (!$job->isInError()) {
                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                } catch (TransactionHistoryException $exception) {
                    $this->skipJob($job, $exception);
                }

                $this->saveJob($job);
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_ORDER_EXPORT_ENABLE)
            && $this->config->isSetFlag(Config::XML_PATH_ORDER_IMPORT_PRODUCTS_RETURNS);
    }

    /**
     * @inheritDoc
     *
     * @throws TransactionHistoryException
     */
    protected function processJob($job)
    {
        $creditmemoEntity = $job->getEntity();

        if ($job->getEntity() instanceof CreditmemoEntity) {
            try {
                $orderSync = $this->getOriginalOrder($creditmemoEntity);
            } catch (NoSuchEntityException $exception) {
                throw new TransactionHistoryException(__(
                    'Order with Fastmag ID #%1 does not seem to come from Magento, skipping.',
                    $creditmemoEntity->getOrderFastmagId()
                ));
            }

            try {
                $order = $this->orderRepository->get($orderSync->getOrderId());
            } catch (NoSuchEntityException $exception) {
                throw new JobException(__(
                    'Order with Magento ID #%1 does not exist in DB, please review it ASAP.',
                    $orderSync->getOrderId()
                ));
            }

            $this->debug(
                $job,
                __('Order #%1 found as original order of the credit memo', $order->getIncrementId())->render()
            );

            if (!$order->canCreditmemo()) {
                throw new JobException(__('Order  #%1 can\'t create a creditmemo.', $order->getIncrementId()));
            }

            $this->createCreditmemo($order, $creditmemoEntity);
        }
    }

    /**
     * Get original order from its Fastmag ID
     *
     * @param CreditmemoEntity $creditmemoEntity
     *
     * @return OrderSync
     *
     * @throws NoSuchEntityException
     */
    protected function getOriginalOrder(CreditmemoEntity $creditmemoEntity): OrderSync
    {
        $orderFastmagId = $creditmemoEntity->getOrderFastmagId();

        return $this->orderSyncRepository->getLastSyncByTransactionId($orderFastmagId);
    }

    /**
     * Create matching creditmemo in Magento for the order
     *
     * @param Order            $order
     * @param CreditmemoEntity $creditmemoEntity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function createCreditmemo(Order $order, CreditmemoEntity $creditmemoEntity): void
    {
        $creditmemoItems = [];
        $creditmemoArguments = $this->creditmemoArgumentsFactory->create();
        $creditmemoArguments->setShippingAmount(0);

        foreach ($order->getItems() as $orderItem) {
            $creditmemoItem = $this->creditmemoItemFactory->create();
            $creditmemoItem->setOrderItemId($orderItem->getItemId())
                ->setQty(0);

            $creditmemoItems[$orderItem->getSku()] = $creditmemoItem;
        }

        foreach ($creditmemoEntity->getItems() as $item) {
            if ($this->isProductShippingRates($item)) {
                $creditmemoArguments->setShippingAmount($order->getBaseShippingAmount());

                $this->debug($this->currentJob, __('Shipping rates added to the credit memo')->render());
            } else {
                $sku = $this->simpleProductSkuGenerator->execute($item->getRef(), $item->getSize(), $item->getColor());

                if (array_key_exists($sku, $creditmemoItems)) {
                    $creditmemoItems[$sku]->setQty($item->getQuantity());

                    $this->debug($this->currentJob, __('Item with SKU "%1" added to the credit memo', $sku)->render());
                } else {
                    throw new JobException(__('Product with SKU "%1" does not exist in the original order', $sku));
                }
            }
        }

        $creditmemoId = $this->refundOrderService->execute(
            $order->getId(),
            $creditmemoItems,
            $this->config->isSetFlag(Config::XML_PATH_ORDER_IMPORT_NOTIFY_CUSTOMERS),
            false,
            null,
            $creditmemoArguments
        );

        $this->debug($this->currentJob, __('Credit memo #%1 created', $creditmemoId)->render());
    }

    /**
     * Check if item is the shipping rate utilitarian product
     *
     * @param ItemEntity $item
     *
     * @return bool
     */
    protected function isProductShippingRates(ItemEntity $item): bool
    {
        return $item->getRef() === Constants::PRODUCT_FP_SKU;
    }
}
