<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2025 HOMEMADE.IO SAS
 * @date      2025-09-03
 ******************************************************************************/

namespace Fastmag\Sync\Model\Rule;

use Exception;
use Fastmag\Sync\Api\Data\Rule\CarrierInterface as Rule;
use Fastmag\Sync\Api\Data\Rule\CarrierInterfaceFactory as RuleFactory;
use Fastmag\Sync\Api\Data\Rule\CarrierSearchResultsInterfaceFactory as SearchResultsFactory;
use Fastmag\Sync\Api\Rule\CarrierRepositoryInterface;
use Fastmag\Sync\Model\ResourceModel\Rule\Carrier as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Rule\Carrier\CollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface as CollectionProcessor;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * Class CarrierRepository
 *
 * Repository class for carrier rules model
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 * phpcs:disable Magento2.Commenting.ClassPropertyPHPDocFormatting.Missing
 */
class CarrierRepository implements CarrierRepositoryInterface
{
    private ResourceModel $resource;
    private RuleFactory $ruleFactory;
    private CollectionFactory $collectionFactory;
    private SearchResultsFactory $searchResultsFactory;

    /** @var CollectionProcessor $collectionProcessor */
    private $collectionProcessor;

    /**
     * Carrier constructor
     *
     * @param ResourceModel            $resource
     * @param RuleFactory              $ruleFactory
     * @param CollectionFactory        $collectionFactory
     * @param SearchResultsFactory     $searchResultsFactory
     * @param CollectionProcessor|null $collectionProcessor
     */
    public function __construct(
        ResourceModel        $resource,
        RuleFactory          $ruleFactory,
        CollectionFactory    $collectionFactory,
        SearchResultsFactory $searchResultsFactory,
        ?CollectionProcessor $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->ruleFactory = $ruleFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(Rule $rule): Rule
    {
        try {
            $this->resource->save($rule);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $rule;
    }

    /**
     * @inheritDoc
     */
    public function getById($ruleId)
    {
        $rule = $this->ruleFactory->create();
        $this->resource->load($rule, $ruleId);
        if (!$rule->getId()) {
            throw new NoSuchEntityException(__('The rule with the "%1" ID doesn\'t exist.', $ruleId));
        }

        return $rule;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(Rule $rule): bool
    {
        try {
            $this->resource->delete($rule);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($ruleId)
    {
        return $this->delete($this->getById($ruleId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessor
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
